/*
 * This is some unsupported source-code to patch graphs which
 * use rrd-files.
 * The rrd-file have to be exported to an xml file before letting
 * this utility modify the data.
 * 
 * Example:
 * rrdtool dump ~maison/rrd/temperature.rrd > t1.xml
 * ./rrd_replace -p -i t1.xml -c fan_out
 * 
 * Look in the scripts to find out how it's used.
 *
 * /Christian Magnusson, mag(at)mag.cx
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/mman.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <ctype.h>

#include "rrd_replace.h"

#define VERSION "1.0"

/*
  This is hardcoded in sensor-id's, column names and column nr
  from the rrd-file.
*/
struct sensor *update_sensor_pointers()
{
  add_sensor("10.061847000800", "heating", 8);
  add_sensor("10.4D8746000800", "foodstorage", 4);
  add_sensor("10.233B40000800", "guestroom_up", 7);
  add_sensor("10.5D1947000800", "kitchen", 9);
  add_sensor("10.1A9246000800", "office", 12);
  add_sensor("10.A22840000800", "sauna", 13);
  add_sensor("10.2BD346000800", "storage", 14);
  add_sensor("10.E54347000800", "tvroom", 15);
  add_sensor("10.B3E166000800", "garage", 5);
  add_sensor("10.AE9866000800", "outdoor", 2);
  add_sensor("10.E2C746000800", "fan_out", 6);

  add_sensor("10.000000000000", "bedroom", 1);
  add_sensor("10.000000000000", "laundry", 10);
  add_sensor("10.000000000000", "fireplace", 3);
  add_sensor("10.000000000000", "livingroom", 11);
  return sensors;
}

void usage(char **argv)
{
  //printf("rrddump_replace orgfile.xml newfile.xml newdata.log\n");
  printf("-p = print values to stdout\n");
  printf("-r = replace value\n");
  printf("-z = clear value to N/A\n");
  printf("%s -r -i orgfile.xml -o newfile.xml -c <column> -d newdata.log\n", argv[0]);
  printf("%s -z -i orgfile.xml -o newfile.xml -c <column> [-t <time>]\n", argv[0]);
  printf("%s -p -i orgfile.xml -c <column> [-t <time>]\n", argv[0]);
  printf("\t<time> =  [-]<time>[-]\n");
  printf("\t<time> =  <time>-<time>\n");
  printf("\t<column> =  <nr|10.000000000|name>\n");
  printf("Version: %s\n", VERSION);
  exit(0);
}

int my_sscanf(char *p, unsigned long *t)
{
#if 0
  /* damn... this was very slow... */
  int rc = sscanf(p, "%ld", t);
  if(rc != 1) return 0;
#else
  *t = atol(p);
#endif
  return 1;
}


unsigned long get_time(char *s)
{
  char *p;
  unsigned long t;
#if 1
  p = strstr(s, "<!--");
  if(!p) return 0;
  p = strstr(p, " / ");
  if(!p) return 0;
  if(my_sscanf(p+3, &t) != 1) return 0;
#else
  p = s;
  while(*p && *p!='<' && *p!='\n') p++;
  if(!*p) return 0;
  p++;
  if(*p != '!') return 0;
  while(*p && *p!='/' && *p!='\n') p++;
  if(!*p) return 0;
  p++;
  if(my_sscanf(p, &t) != 1) return 0;
#endif
  return t;
}

int get_section(char *p, char **data_start, char **data_end, char **db_start, char **db_end)
{
  if(!data_start || !data_end || !db_start || !db_end) {
    printf("get_section: some pointer is NULL\n");
    return -1;
  }
  *data_start = strstr(p, "<rra>");
  if(!*data_start) {
    printf("can't find start\n");
    return -1;
  }
  *data_end = strstr(*data_start, "</rra>");
  if(!*data_end) {
    printf("can't find end\n");
    return -1;
  }
  
  *db_start = strstr(*data_start, "<database>");
  if(!*db_start) {
    printf("can't find dbstart\n");
    return -1;
  }
  *db_end = strstr(*db_start, "</database>");
  if(!*db_end) {
    printf("can't find dbend\n");
    return -1;
  }
  return 0;
}

#define BUFSIZE 8192

struct temp *old_temps = NULL;

int change_data(char *p, char **np, int column)
{
  char *s;
  //  int len;
  char *buf;
  unsigned long rowtime;
  char *data_start;
  char *data_end;
  char *db_start;
  char *db_end;
  struct temp *oldt;

  if(!p) return -1;
  if(!np || !*np) return -1;

#if 0
  buf = (char *)malloc(BUFSIZE);
  if(!buf) {
    printf("error malloc\n");
    return -1;
  }
#endif

  if(get_section(p, &data_start, &data_end, &db_start, &db_end) == -1) {
    return -1;
  }

  /* used for next call to change_data */
  *np = db_end;

  oldt = old_temps;

  if(oldt->column != column) {
    printf("Error: Column differ from datafile %d!=%d\n", oldt->column, column);
    exit(0);
    //return -1;
  }


  s = db_start;
  buf = s;
  while(s < db_end) {
#if 0
    rc = sscanf(s, "%[^\n]\n", buf);
    len = strlen(buf);
#else
    buf = s;
#endif
    //printf("rc=%d  len=%d  buf=[%s]\n", rc, len, buf);
    rowtime = get_time(buf);
    if(rowtime > 0) {
      //printf("found rowtime=%ld\n", rowtime);

      if(oldt) {
	//printf("found rowtime=%ld  oldt=%ld\n", rowtime, oldt->time);
	oldt->avg = oldt->temp;

	if(oldt->time < rowtime) {
	  int nr = 0;
	  float sum = 0;
	  //printf("really old values %ld < %ld\n", oldt->time, rowtime);
	  while(oldt) {
	    sum += oldt->temp;
	    nr++;
	    if(oldt->time >= rowtime) {
	      //printf("fast forward to %ld\n", oldt->time);
	      // calculate average
	      oldt->avg = sum/(float)nr;
	      break;
	    }
	    oldt = oldt->next;
	  }
	  if(!oldt) {
	    printf("end of values\n");
	    goto cleanup;
	  }
	}
	if(rowtime == oldt->time) {
	  char *a, *b;
	  int len1, len2;
	  int i;

	  //printf("found rowtime=%ld\n", rowtime);
	  a = buf;
	  for(i=0; i<oldt->column; i++) {
	    a = strstr(a+1, "<v>");
	    if(!a) {
	      printf("error finding %d <v>\n", i);
	      break;
	    }
	  }
	  if(a) {
	    char tmp[512];
	    int vpos;
	    vpos = a-buf;
	    b = strstr(a, "</v>");
	    if(b) {
	      b += strlen("</v>");
	      len1 = b-a;
	      strncpy(tmp, a, len1);
	      tmp[len1] = '\0';

	      if(len1 < 24) {
		//printf("bad length??? len1=%d [%s]\n", len1, tmp);
		//len2 = sprintf(tmp, "<v> %d ", (unsigned int)oldt->temp);
#ifdef AVG
		len2 = sprintf(tmp, "<v>%5.1f", oldt->avg);
#else
		len2 = sprintf(tmp, "<v>%5.1f", oldt->temp);
#endif
	      } else {
		//printf("replace (%d)[%s] ", len1, tmp);
#ifdef AVG
		len2 = sprintf(tmp, "<v> %e ", oldt->avg);
#else
		len2 = sprintf(tmp, "<v> %e ", oldt->temp);
#endif
		//printf("to (%d)[%s] ", len2, tmp);
	      }
	      while(len2 < len1-4) {
		tmp[len2++] = ' ';
	      }
	      memcpy(&tmp[len2], "</v>", 4);
	      len2 += 4;
	      tmp[len2] = '\0';
	      /* replace original row */
	      memcpy(&s[vpos], tmp, strlen(tmp));
	    } else {
	      printf("can't find </v>\n");
	    }
    
	  }
	  oldt = oldt->next;
	}
      } else {
	printf("no more old temps available\n");
	return 0;
      }
    }
#if 0
    s += len + 1;  /* carrige return */
#else
    while(*s && *s!='\n') s++;
    s++;
#endif
  }
 cleanup:
  return 0;
}


int print_data(char *p, char **np, time_t tm_from, time_t tm_to, int column)
{
  char *s;
  char *buf;
  unsigned long rowtime;
  char *data_start;
  char *data_end;
  char *db_start;
  char *db_end;

  if(!p) return -1;
  if(!np || !*np) return -1;

  if(get_section(p, &data_start, &data_end, &db_start, &db_end) == -1) {
    return -1;
  }

  /* used for next call to change_data */
  *np = db_end;

  s = db_start;
  buf = s;
  while(s < db_end) {
#if 0
    rc = sscanf(s, "%[^\n]\n", buf);
    len = strlen(buf);
#else
    buf = s;
#endif
    //printf("rc=%d  len=%d  buf=[%s]\n", rc, len, buf);
    rowtime = get_time(buf);
    if(rowtime > 0) {
      //printf("found rowtime=%ld\n", rowtime);

      if((rowtime >= tm_from) && (rowtime <= tm_to)) {
	char *a, *b;
	int len1;
	int i;

	//printf("found rowtime=%ld\n", rowtime);
	a = buf;
	for(i=0; i<column; i++) {
	  a = strstr(a+1, "<v>");
	  if(!a) {
	    printf("error finding %d <v>\n", i);
	    break;
	  }
	}
	if(a) {
	  char tmp[512];
	  int vpos;
	  vpos = a-buf;
	  b = strstr(a, "</v>");
	  if(b) {
	    b += strlen("</v>");
	    len1 = b-a;
	    strncpy(tmp, a, len1);
	    tmp[len1] = '\0';

	    {
	      float f;
	      sscanf(tmp+4, "%e", &f);
	      printf("%ld:%5.1f\n", rowtime, f);
	    }
	  } else {
	    printf("can't find </v>\n");
	  }
    
	}
      }
    }
#if 0
    s += len + 1;  /* carrige return */
#else
    while(*s && *s!='\n') s++;
    s++;
#endif
  }
  return 0;
}

int clear_data(char *p, char **np, time_t tm_from, time_t tm_to, int column)
{
  char *s;
  char *buf;
  unsigned long rowtime;
  char *data_start;
  char *data_end;
  char *db_start;
  char *db_end;
  //struct temp *t;
  //struct temp *oldt;

  if(!p) return -1;
  if(!np || !*np) return -1;

#if 0
  buf = (char *)malloc(BUFSIZE);
  if(!buf) {
    printf("error malloc\n");
    return -1;
  }
#endif

  if(get_section(p, &data_start, &data_end, &db_start, &db_end) == -1) {
    return -1;
  }

#if 0
  t = (struct temp *)malloc(sizeof(struct temp));
  if(!t) {
    printf("Error malloc\n");
    return -1;
  }
  memset(t, 0, sizeof(struct temp));
  t->time = tm_from;
  t->temp = -1;
  t->column = -1;
  old_temps = t;

  t = (struct temp *)malloc(sizeof(struct temp));
  if(!t) {
    printf("Error malloc\n");
    return -1;
  }
  memset(t, 0, sizeof(struct temp));
  t->time = tm_to;
  t->temp = -1;
  t->column = -1;
  old_temps->next = t;
#endif

  /* used for next call to change_data */
  *np = db_end;

  s = db_start;
  buf = s;
  while(s < db_end) {
#if 0
    rc = sscanf(s, "%[^\n]\n", buf);
    len = strlen(buf);
#else
    buf = s;
#endif
    //printf("rc=%d  len=%d  buf=[%s]\n", rc, len, buf);
    rowtime = get_time(buf);
    if(rowtime > 0) {
      //printf("found rowtime=%ld\n", rowtime);

      if((rowtime >= tm_from) && (rowtime <= tm_to)) {
	char *a, *b;
	int len1, len2;
	int i;

	//printf("found rowtime=%ld\n", rowtime);
	a = buf;
	for(i=0; i<column; i++) {
	  a = strstr(a+1, "<v>");
	  if(!a) {
	    printf("error finding %d <v>\n", i);
	    break;
	  }
	}
	if(a) {
	  char tmp[512];
	  int vpos;
	  vpos = a-buf;
	  b = strstr(a, "</v>");
	  if(b) {
	    b += strlen("</v>");
	    len1 = b-a;
	    strncpy(tmp, a, len1);
	    tmp[len1] = '\0';

	    len2 = sprintf(tmp, "<v> NaN");

	    while(len2 < len1-4) {
	      tmp[len2++] = ' ';
	    }
	    memcpy(&tmp[len2], "</v>", 4);
	    len2 += 4;
	    tmp[len2] = '\0';
	    /* replace original row */
	    memcpy(&s[vpos], tmp, strlen(tmp));
	  } else {
	    printf("can't find </v>\n");
	  }
    
	}
      }
    }
#if 0
    s += len + 1;  /* carrige return */
#else
    while(*s && *s!='\n') s++;
    s++;
#endif
  }
  return 0;
}


struct sensor *sensors = NULL;

struct sensor *add_sensor(char *name, char *location, int column)
{
  struct sensor *s, *ss;
  s = (struct sensor *)malloc(sizeof(struct sensor));
  memset(s, 0, sizeof(struct sensor));
  strcpy(s->id, name);
  strcpy(s->location, location);
  s->column = column;
  
  if(!sensors) {
    sensors = s;
  } else {
    ss = sensors;
    while(ss->next) {
      ss = ss->next;
    }
    ss->next = s;
  }
  return s;
}

int print_columns()
{
  struct sensor *s;
  if(!sensors) {
    sensors = update_sensor_pointers();
    if(!sensors) exit(0);
  }

  s = sensors;
  while(s) {
    printf("Column %3d [%15s] [%14s]\n", s->column, s->id, s->location);
    s = s->next;
  }
  return 0;
}

int get_sensor_column(char *file)
{
  struct sensor *s;
  if(!sensors) {
    sensors = update_sensor_pointers();
    if(!sensors) exit(0);
  }
  if(!file) return -1;

  s = sensors;
  while(s) {
    if(strstr(file, s->id)) {
      return s->column;
    }
    s = s->next;
  }
  return -1;
}

int get_location_column(char *location)
{
  struct sensor *s;
  if(!sensors) {
    sensors = update_sensor_pointers();
    if(!sensors) exit(0);
  }
  if(!location) return -1;

  s = sensors;
  while(s) {
    if(strstr(location, s->location)) {
      return s->column;
    }
    s = s->next;
  }
  return -1;
}

int get_column_name(int column_nr, char *column_name)
{
  struct sensor *s;
  if(!sensors) {
    sensors = update_sensor_pointers();
    if(!sensors) exit(0);
  }

  s = sensors;
  while(s) {
    if(s->column == column_nr) {
      strcpy(column_name, s->location);
      return 1;
    }
    s = s->next;
  }
  return -1;
}


int get_column_sensid(char *sensid)
{
  struct sensor *s;
  if(!sensors) {
    sensors = update_sensor_pointers();
    if(!sensors) exit(0);
  }

  s = sensors;
  while(s) {
    if(strcmp(s->id, sensid) == 0) {
      return s->column;
    }
    s = s->next;
  }
  return -1;
}

struct temp *last_temp = NULL;

int load_temp(char *file)
{
  int rc;
  int nr;
  FILE *fp;
  char buf[80];
  unsigned long mintime;
  unsigned long lasttime;
  unsigned long maxtime;
  float newtemp;
  unsigned long newtime;
  struct temp *t;
  int column;
  struct tm *tt;
  char tmp[80];

  column = get_sensor_column(file);
  if(column < 0) {
    printf("can't find column for sensor [%s]\n", file);
    return -1;
  }

  fp = fopen(file, "r");
  if(!fp) {
    printf("error open temp-log\n");
    return -1;
  }

  rc = 0;
  nr = 0;
  mintime = -1;
  maxtime = 0;
  lasttime = 0;
  while(!feof(fp)) {
    if(fgets(buf, 80, fp) == NULL) {
      if(feof(fp)) {
	//printf("end of file\n");
      } else {
	printf("error read\n");
	rc = -1;
      }
      break;
    }
    if(sscanf(buf, "%ld:%f", &newtime, &newtemp) != 2) {
      printf("error parse [%s]\n", buf);
      rc = -1;
      break;
    }
    if(newtime <= lasttime) {
      printf("newtime(%ld) is not bigger than lasttime(%ld) [%s]\n", newtime, lasttime, buf);
      rc = -1;
      break;
    }
    lasttime = newtime;
    if(newtime < mintime) mintime = newtime;
    if(newtime > maxtime) maxtime = newtime;

    t = (struct temp *)malloc(sizeof(struct temp));
    if(!t) {
      printf("Error malloc\n");
      rc = -1;
      break;
    }
    memset(t, 0, sizeof(struct temp));
    t->time = newtime;
    t->temp = newtemp;
    t->column = column;
    t->prev = last_temp;
    if(last_temp) {
      last_temp->next = t;
    } else {
      old_temps = t;
    }
    last_temp = t;
    nr++;
  }
  fclose(fp);
  fp = NULL;
  printf("loaded %d temps from file\n", nr);
  
  tt = localtime(&mintime);
  rc = strftime(tmp, 80, "%a %b %d %H:%M:%S %Y", tt);
  printf("mintime = %ld  (%s)\n", mintime, tmp);

  tt = localtime(&maxtime);
  rc = strftime(tmp, 80, "%a %b %d %H:%M:%S %Y", tt);
  printf("maxtime = %ld  (%s)\n", maxtime, tmp);
  //printf("test = %e\n", 0.75);
  //printf("test = %e\n", 25.8725);
  return rc;
}


void replace(char *input_file, char *output_file, char *data_file, int column)
{
  int bytes, rc;
  char *p;
  FILE *fp = NULL, *fp2 = NULL;
  char *buf = NULL;
  char *np, *tp;
  unsigned long slen;
  int pos, len;

  fp = fopen(input_file, "r");
  if(!fp) {
    printf("can't read rrd-file [%s]\n", input_file);
    goto cleanup;
  }
  fp2 = fopen(output_file, "w");
  if(!fp2) {
    printf("can't write rrd-file [%s]\n", output_file);
    goto cleanup;
  }

  rc = fseek(fp, 0, SEEK_END);
  if(rc < 0) {
    printf("fseek error\n");
    goto cleanup;
  }
  slen = ftell(fp);
  if(slen < 0 || fseek(fp, 0, SEEK_SET)<0) {
    printf("fseek error\n");
    goto cleanup;
  }
  buf = (char *)malloc(slen);
  if(!buf) {
    printf("error malloc %ld\n", slen);
    goto cleanup;
  }

  bytes = 0;
  while(!feof(fp)) {
    rc = fread(&buf[bytes], 1, BUFSIZE, fp);
    if(rc <= 0) {
      printf("error read rc=%d\n", rc);
      if(ferror(fp)) {
	printf("file error\n");
	rc = -1;
      }
      if(feof(fp)) {
	//printf("end of file reached\n");
	rc = 0;
      }
      break;
    }
    bytes += rc;
  }
  fclose(fp);
  fp = NULL;

  if(bytes != slen) {
    printf("%d != %ld\n", bytes, slen);
    goto cleanup;
  }
  //printf("%d bytes read\n", bytes);

#if 0
  {
    char tmp[128];
    printf("file begin with: \n");
    strncpy(tmp, buf, 80);
    tmp[80] = '\0';
    printf("[%s]\n", tmp);
  }
#endif

  rc = load_temp(data_file);
  if(rc < 0) {
    goto cleanup;
  }
  
  p = buf;

  printf("Update AVERAGE 60\n");
  tp = p;
  if(change_data(tp, &np, column) == -1) {
    printf("error update AVERAGE 60\n");
    goto cleanup;
  }
  printf("Update AVERAGE 1800\n");
  tp = np;
  if(change_data(tp, &np, column) == -1) {
    printf("error update AVERAGE 1800\n");
    goto cleanup;
  }
  printf("Update AVERAGE 7200\n");
  tp = np;
  if(change_data(tp, &np, column) == -1) {
    printf("error update AVERAGE 7200\n");
    goto cleanup;
  }
#if 1
  printf("Update AVERAGE 86400\n");
  tp = np;
  if(change_data(tp, &np, column) == -1) {
    printf("error update AVERAGE 86400\n");
    goto cleanup;
  }
  tp = np;
#endif

  pos = 0;
  while(pos < slen) {
    len = ((slen-pos > BUFSIZE) ? BUFSIZE : slen-pos);
    rc = fwrite(&buf[pos], 1, len, fp2);
    if(rc <= 0) {
      printf("error write rc=%d\n", rc);
      if(ferror(fp2)) {
	printf("file error\n");
	rc = -1;
      }
      break;
    }
    pos += rc;
  }
  fclose(fp2);
  fp2 = NULL;

 cleanup:
  if(fp) fclose(fp);
  if(fp2) fclose(fp2);
  if(buf) free(buf);
}

void debug_print(char *input_file, time_t tm_from, time_t tm_to, int column)
{
  char *p;
  int bytes, rc;
  FILE *fp = NULL;
  char *buf = NULL;
  char *np, *tp;
  unsigned long slen;

  fp = fopen(input_file, "r");
  if(!fp) {
    printf("can't read rrd-file [%s]\n", input_file);
    goto cleanup;
  }

  rc = fseek(fp, 0, SEEK_END);
  if(rc < 0) {
    printf("fseek error\n");
    goto cleanup;
  }
  slen = ftell(fp);
  if(slen < 0 || fseek(fp, 0, SEEK_SET)<0) {
    printf("fseek error\n");
    goto cleanup;
  }
  buf = (char *)malloc(slen);
  if(!buf) {
    printf("error malloc %ld\n", slen);
    goto cleanup;
  }

  bytes = 0;
  while(!feof(fp)) {
    rc = fread(&buf[bytes], 1, BUFSIZE, fp);
    if(rc <= 0) {
      printf("error read rc=%d\n", rc);
      if(ferror(fp)) {
	printf("file error\n");
	rc = -1;
      }
      if(feof(fp)) {
	//printf("end of file reached\n");
	rc = 0;
      }
      break;
    }
    bytes += rc;
  }
  fclose(fp);
  fp = NULL;

  if(bytes != slen) {
    printf("%d != %ld\n", bytes, slen);
    goto cleanup;
  }
  //printf("%d bytes read\n", bytes);

  p = buf;

  tp = p;
  printf("Print AVERAGE 60\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print AVERAGE 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Print AVERAGE 1800\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print AVERAGE 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Print AVERAGE 7200\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print AVERAGE 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Print AVERAGE 86400\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print AVERAGE 86400\n");
    goto cleanup;
  }
  tp = np;



  printf("Print MIN 60\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MIN 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MIN 1800\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MIN 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MIN 7200\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MIN 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MIN 86400\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MIN 86400\n");
    goto cleanup;
  }
  tp = np;



  printf("Print MAX 60\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MAX 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MAX 1800\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MAX 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MAX 7200\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MAX 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MAX 86400\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MAX 86400\n");
    goto cleanup;
  }
  tp = np;


 cleanup:
  if(fp) fclose(fp);
  if(buf) free(buf);
}

void zerovalue(char *input_file, char *output_file, time_t tm_from, time_t tm_to, int column)
{
  char *p;
  int bytes, rc;
  FILE *fp = NULL, *fp2 = NULL;
  char *buf = NULL;
  char *np, *tp;
  unsigned long slen;
  int pos, len;

  fp = fopen(input_file, "r");
  if(!fp) {
    printf("can't read rrd-file [%s]\n", input_file);
    goto cleanup;
  }
  fp2 = fopen(output_file, "w");
  if(!fp2) {
    printf("can't write rrd-file [%s]\n", output_file);
    goto cleanup;
  }

  rc = fseek(fp, 0, SEEK_END);
  if(rc < 0) {
    printf("fseek error\n");
    goto cleanup;
  }
  slen = ftell(fp);
  if(slen < 0 || fseek(fp, 0, SEEK_SET)<0) {
    printf("fseek error\n");
    goto cleanup;
  }
  buf = (char *)malloc(slen);
  if(!buf) {
    printf("error malloc %ld\n", slen);
    goto cleanup;
  }

  bytes = 0;
  while(!feof(fp)) {
    rc = fread(&buf[bytes], 1, BUFSIZE, fp);
    if(rc <= 0) {
      printf("error read rc=%d\n", rc);
      if(ferror(fp)) {
	printf("file error\n");
	rc = -1;
      }
      if(feof(fp)) {
	//printf("end of file reached\n");
	rc = 0;
      }
      break;
    }
    bytes += rc;
  }
  fclose(fp);
  fp = NULL;

  if(bytes != slen) {
    printf("%d != %ld\n", bytes, slen);
    goto cleanup;
  }
  //printf("%d bytes read\n", bytes);

  p = buf;

  tp = p;
  printf("Clear AVERAGE 60\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear AVERAGE 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear AVERAGE 1800\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear AVERAGE 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear AVERAGE 7200\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear AVERAGE 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear AVERAGE 86400\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear AVERAGE 86400\n");
    goto cleanup;
  }
  tp = np;



  printf("Clear MIN 60\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MIN 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MIN 1800\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MIN 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MIN 7200\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MIN 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MIN 86400\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MIN 86400\n");
    goto cleanup;
  }
  tp = np;



  printf("Clear MAX 60\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MAX 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MAX 1800\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MAX 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MAX 7200\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MAX 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MAX 86400\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MAX 86400\n");
    goto cleanup;
  }
  tp = np;


  pos = 0;
  while(pos < slen) {
    len = ((slen-pos > BUFSIZE) ? BUFSIZE : slen-pos);
    rc = fwrite(&buf[pos], 1, len, fp2);
    if(rc <= 0) {
      printf("error write rc=%d\n", rc);
      if(ferror(fp2)) {
	printf("file error\n");
	rc = -1;
      }
      break;
    }
    pos += rc;
  }
  fclose(fp2);
  fp2 = NULL;

 cleanup:
  if(fp) fclose(fp);
  if(fp2) fclose(fp2);
  if(buf) free(buf);
}

int main(int argc, char **argv)
{
  char *np;
  int len;
  int n;
  char column_name[80] = "\000";
  int column_nr = -1;
  int mode_print = 0;
  int mode_zero = 0;
  int mode_replace = 0;
  time_t tm_from = 0, tm_to = 0;
  char input_file[128] = "\000";
  char output_file[128] = "\000";
  char data_file[128] = "\000";

  n = 1;
  while(n<argc) {
    switch(argv[n][0]) {
    case '-':
      switch(argv[n++][1]) {
      case 'r':
	mode_replace = 1;
	break;
      case 'z':
	mode_zero = 1;
	break;
      case 'p':
	mode_print = 1;
	break;
      case 'c':
	if(n>=argc) usage(argv);
	if(isdigit(argv[n][0])) {
	  if((strlen(argv[n])>12) && argv[n][2]=='.') {
	    column_nr = get_column_sensid(argv[n]);
	    if(column_nr < 0) {
	      printf("Can't find column nr %s\n", argv[n]);
	      column_nr = -1;
	    } else {
	      get_column_name(column_nr, column_name);
	    }
	  } else {
	    column_nr = atoi(argv[n]);
	    if(get_column_name(column_nr, column_name) < 0) {
	      printf("Can't find column nr %d\n", column_nr);
	      column_nr = -1;
	    }
	  }
	} else {
	  strcpy(column_name, argv[n]);
	  column_nr = get_location_column(argv[n]);
	  if(column_nr < 0) {
	    printf("Can't find column [%s]\n", column_name);
	    column_nr = -1;
	  }
	}
	
	n++;
	break;
      case 'i':
	if(n>=argc) usage(argv);
	strcpy(input_file, argv[n]);
	n++;
	break;
      case 'o':
	if(n>=argc) usage(argv);
	strcpy(output_file, argv[n]);
	n++;
	break;
      case 'd':
	if(n>=argc) usage(argv);
	strcpy(data_file, argv[n]);
	n++;
	break;
      case 't':
	if(n>=argc) usage(argv);
	np = argv[n];
	len = strlen(np);
	if(np[0] == '-') {
	  tm_from = 0;
	  tm_to = atol(&np[1]);
	} else if(np[len-1] == '-') {
	  tm_from = atol(np);
	  tm_to = (unsigned long)-1;
	} else if(strchr(np, '-')) {
	  tm_from = atol(np);
	  tm_to = atol(strchr(np, '-')+1);
	  if(tm_from > tm_to) {
	    printf("from-time is greater than to-time\n");
	    usage(argv);
	  }
	} else {
	  usage(argv);
	}
	n++;
	break;
      default:
	usage(argv);
	break;
      }
      break;
    default:
      usage(argv);
    }
  }
  if(!mode_replace && !mode_zero && !mode_print) {
    printf("-r, -z or -p not specified.\n");
    usage(argv);
  }
  if(mode_print) {
    if(!input_file[0]) {
      printf("no input-file\n");
      usage(argv);
    }
    if(!tm_from && !tm_to) {
      tm_to = ((time_t)-1);
    }
    if(column_nr < 0) {
      printf("column not found. Available columns are:\n");
      print_columns();
      usage(argv);
    }
    debug_print(input_file, tm_from, tm_to, column_nr);
    exit(0);
  }
  if(mode_replace) {
    if(!input_file[0]) {
      printf("no input-file\n");
      usage(argv);
    }
    if(!output_file[0]) {
      printf("no output-file\n");
      usage(argv);
    }
    if(!data_file[0]) {
      printf("no data-file\n");
      usage(argv);
    }
    if(column_nr < 0) {
      printf("column not found. Available columns are:\n");
      print_columns();
      usage(argv);
    }
    replace(input_file, output_file, data_file, column_nr);
    exit(0);
  }
  if(mode_zero) {
    if(!input_file[0]) {
      printf("no input-file\n");
      usage(argv);
    }
    if(!output_file[0]) {
      printf("no output-file\n");
      usage(argv);
    }
    if(!tm_from && !tm_to) {
      tm_to = ((time_t)-1);
    }
    if(column_nr < 0) {
      printf("column not found. Available columns are:\n");
      print_columns();
      usage(argv);
    }
    zerovalue(input_file, output_file, tm_from, tm_to, column_nr);
  }
  exit(0);
}

