/*
 * This is some unsupported source-code to patch graphs which
 * use rrd-files.
 * The rrd-file have to be exported to an xml file before letting
 * this utility modify the data.
 * 
 * Example:
 * rrdtool dump ~maison/rrd/temperature.rrd > t1.xml
 * ./rrd_replace -p -i t1.xml -c fan_out
 * 
 * Look in the scripts to find out how it's used.
 *
 * /Christian Magnusson, mag(at)mag.cx
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/mman.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <ctype.h>

#include <config.h>
#include <sensor.h>

#define BUFSIZE 8192

char owshell_path[FILENAME_LEN+1] = "";
char owshell_params[FILENAME_LEN+1] = "";

/*
  Try to restore average values in rrd-file when week,month,year
  data will be inserted
*/
#define AVG 1

struct temp {
  unsigned long time;
  float temp;
  float avg;
  int column;
  struct temp *next;
  struct temp *prev;
};

int get_sensors_from_file(char *input_file);
int add_column(int column_nr, char *column_name, char *input_file, char *output_file);
int delete_column(int column_nr, char *column_name, char *input_file, char *output_file);

void usage(char **argv)
{
  printf("\nUsage: %s {-p|-r|-z|-a|-d } -i orgfile.xml ...\n", argv[0]);
  printf("-a = add sensor to file\n");
  printf("-d = delete sensor to file\n");
  printf("-p = print values to stdout\n");
  printf("-r = replace value (with data from log-file)\n");
  printf("-z = clear value to N/A\n");
  printf("%s -a -i orgfile.xml -o newfile.xml -c <column>\n", argv[0]);
  printf("%s -d -i orgfile.xml -o newfile.xml -c <column>\n", argv[0]);
  printf("%s -r -i orgfile.xml -o newfile.xml -c <column> -l newdata.log\n", argv[0]);
  printf("%s -z -i orgfile.xml -o newfile.xml -c <column> [-t <time>]\n", argv[0]);
  printf("%s -p -i orgfile.xml -c <column> [-t <time>]\n", argv[0]);
  printf("\t<time> =  [-]<time>[-]\n");
  printf("\t<time> =  <time>-<time>\n");
  printf("\t<column> =  <nr|10.000000000|name>\n");
  printf("Version: %s\n", VERSION);
  exit(1);
}

int my_sscanf(char *p, unsigned long *t)
{
#if 0
  /* damn... this was very slow... */
  int rc = sscanf(p, "%ld", t);
  if(rc != 1) return 0;
#else
  *t = atol(p);
#endif
  return 1;
}


unsigned long get_time(char *s)
{
  char *p;
  unsigned long t;
#if 1
  p = strstr(s, "<!--");
  if(!p) return 0;
  p = strstr(p, " / ");
  if(!p) return 0;
  if(my_sscanf(p+3, &t) != 1) return 0;
#else
  p = s;
  while(*p && *p!='<' && *p!='\n') p++;
  if(!*p) return 0;
  p++;
  if(*p != '!') return 0;
  while(*p && *p!='/' && *p!='\n') p++;
  if(!*p) return 0;
  p++;
  if(my_sscanf(p, &t) != 1) return 0;
#endif
  return t;
}

int get_section_name(char *p, char *name_start, char *name_end, char **data_start, char **data_end)
{
  if(!data_start || !data_end) {
    printf("get_section: some pointer is NULL\n");
    return -1;
  }
  *data_start = strstr(p, name_start);
  if(!*data_start) {
    //printf("can't find start\n");
    return -1;
  }
  *data_end = strstr(*data_start, name_end);
  if(!*data_end) {
    printf("can't find end\n");
    return -1;
  }
  return 0;
}

struct temp *old_temps = NULL;

#if 1
struct temp *last_temp = NULL;

int get_sensor_column(char *file)
{
  struct sensor *s;
  if(!sensors) {
    printf("No sensors found yet\n");
    exit(0);
  }
  if(!file) return -1;

  s = sensors;
  while(s) {
    if(strstr(file, s->id)) {
      return s->column;
    }
    s = s->next;
  }
  return -1;
}

int load_temp(char *file)
{
  int rc;
  int nr;
  FILE *fp;
  char buf[80];
  unsigned long mintime;
  unsigned long lasttime;
  unsigned long maxtime;
  float newtemp;
  unsigned long newtime;
  struct temp *t;
  int column;
  struct tm *tt;
  char tmp[80];

  column = get_sensor_column(file);
  if(column < 0) {
    printf("can't find column for sensor [%s]\n", file);
    return -1;
  }
  printf("load_temp: column_nr=%d file=%s\n", column, file);

  fp = fopen(file, "r");
  if(!fp) {
    printf("error open temp-log\n");
    return -1;
  }

  rc = 0;
  nr = 0;
  mintime = 0;
  maxtime = 0;
  lasttime = 0;
  while(!feof(fp)) {
    if(fgets(buf, 80, fp) == NULL) {
      if(feof(fp)) {
	//printf("end of file\n");
      } else {
	printf("error read\n");
	rc = -1;
      }
      break;
    }
    if(sscanf(buf, "%ld:%f", &newtime, &newtemp) != 2) {
      printf("error parse [%s]\n", buf);
      rc = -1;
      break;
    }
    if(newtime <= lasttime) {
      printf("newtime(%ld) is not bigger than lasttime(%ld) [%s]\n", newtime, lasttime, buf);
      rc = -1;
      break;
    }
    lasttime = newtime;
    if(newtime < mintime) mintime = newtime;
    if(newtime > maxtime) maxtime = newtime;

    t = (struct temp *)malloc(sizeof(struct temp));
    if(!t) {
      printf("Error malloc\n");
      rc = -1;
      break;
    }
    memset(t, 0, sizeof(struct temp));
    t->time = newtime;
    t->temp = newtemp;
    t->column = column;
    t->prev = last_temp;
    if(last_temp) {
      last_temp->next = t;
    } else {
      old_temps = t;
    }
    last_temp = t;
    nr++;
  }
  fclose(fp);
  fp = NULL;
  printf("loaded %d temps from file\n", nr);
  
  tt = localtime(&mintime);
  rc = strftime(tmp, 80, "%a %b %d %H:%M:%S %Y", tt);
  printf("mintime = %ld  (%s)\n", mintime, tmp);

  tt = localtime(&maxtime);
  rc = strftime(tmp, 80, "%a %b %d %H:%M:%S %Y", tt);
  printf("maxtime = %ld  (%s)\n", maxtime, tmp);
  //printf("test = %e\n", 0.75);
  //printf("test = %e\n", 25.8725);
  return rc;
}
#endif

int change_data(char *p, char **np, int column)
{
  char *s;
  //  int len;
  char *buf;
  unsigned long rowtime;
  char *data_start;
  char *data_end;
  char *db_start;
  char *db_end;
  struct temp *oldt;

  if(!p) return -1;
  if(!np || !*np) return -1;

#if 0
  buf = (char *)malloc(BUFSIZE);
  if(!buf) {
    printf("error malloc\n");
    return -1;
  }
#endif

  if(get_section_name(p, "<rra>", "</rra>",  &data_start, &data_end) == -1) {
    printf("can't find <rra> tags\n");
    return -1;
  }
  if(get_section_name(data_start, "<database>", "</database>",  &db_start, &db_end) == -1) {
    printf("can't find <database> tags\n");
    return -1;
  }

  /* used for next call to change_data */
  *np = db_end;

  if(!old_temps) {
    printf("error: old_temps == NULL\n");
    return -1;
  }
  oldt = old_temps;

  if(oldt->column != column) {
    printf("Error: Column differ from datafile %d!=%d\n", oldt->column, column);
    exit(0);
    //return -1;
  }


  s = db_start;
  buf = s;
  while(s < db_end) {
#if 0
    rc = sscanf(s, "%[^\n]\n", buf);
    len = strlen(buf);
#else
    buf = s;
#endif
    //printf("rc=%d  len=%d  buf=[%s]\n", rc, len, buf);
    rowtime = get_time(buf);
    //printf("rowtime=%d\n",rowtime);
    if(rowtime > 0) {
      //printf("found rowtime=%ld\n", rowtime);

      if(oldt) {
	//printf("found rowtime=%ld  oldt=%ld\n", rowtime, oldt->time);
	oldt->avg = oldt->temp;

	if(oldt->time < rowtime) {
	  int nr = 0;
	  float sum = 0;
	  //printf("really old values %ld < %ld\n", oldt->time, rowtime);
	  while(oldt) {
	    sum += oldt->temp;
	    nr++;
	    if(oldt->time >= rowtime) {
	      //printf("fast forward to %ld\n", oldt->time);
	      // calculate average
	      oldt->avg = sum/(float)nr;
	      break;
	    }
	    oldt = oldt->next;
	  }
	  if(!oldt) {
	    printf("end of values\n");
	    goto cleanup;
	  }
	}
	if(rowtime == oldt->time) {
	  char *a, *b;
	  int len1, len2;
	  int i;

	  //printf("found rowtime=%ld\n", rowtime);
	  a = buf;
	  for(i=0; i<=oldt->column; i++) {
	    a = strstr(a+1, "<v>");
	    if(!a) {
	      printf("error finding %d <v>\n", i);
	      break;
	    }
	  }
	  if(a) {
	    char tmp[512];
	    int vpos;
	    vpos = a-buf;
	    b = strstr(a, "</v>");
	    if(b) {
	      b += strlen("</v>");
	      len1 = b-a;
	      strncpy(tmp, a, len1);
	      tmp[len1] = '\0';

	      if(len1 < 24) {
		//printf("bad length??? len1=%d [%s]\n", len1, tmp);
		//len2 = sprintf(tmp, "<v> %d ", (unsigned int)oldt->temp);
#ifdef AVG
		len2 = sprintf(tmp, "<v>%5.1f", oldt->avg);
#else
		len2 = sprintf(tmp, "<v>%5.1f", oldt->temp);
#endif
	      } else {
		printf("replace (%d)[%s] ", len1, tmp);
#ifdef AVG
		len2 = sprintf(tmp, "<v> %e ", oldt->avg);
#else
		len2 = sprintf(tmp, "<v> %e ", oldt->temp);
#endif
		//printf("to (%d)[%s] ", len2, tmp);
	      }
	      while(len2 < len1-4) {
		tmp[len2++] = ' ';
	      }
	      memcpy(&tmp[len2], "</v>", 4);
	      len2 += 4;
	      tmp[len2] = '\0';
	      /* replace original row */
	      memcpy(&s[vpos], tmp, strlen(tmp));
	    } else {
	      printf("can't find </v>\n");
	    }
    
	  }
	  oldt = oldt->next;
	}
      } else {
	printf("no more old temps available\n");
	return 0;
      }
    }
#if 0
    s += len + 1;  /* carrige return */
#else
    while(*s && *s!='\n') s++;
    s++;
#endif
  }
 cleanup:
  return 0;
}

int add_column(int column_nr, char *column_name, char *input_file, char *output_file)
{
  char ds_sect[] = "\
        <ds>\n\
                <name> %s </name>\n\
                <type> GAUGE </type>\n\
                <minimal_heartbeat> 120 </minimal_heartbeat>\n\
                <min> -3.9900000000e+01 </min>\n\
                <max> 9.9000000000e+01 </max>\n\
\n\
                <!-- PDP Status -->\n\
                <last_ds> UNKN </last_ds>\n\
                <value> 0.0000000000e+00 </value>\n\
                <unknown_sec> 0 </unknown_sec>\n\
        </ds>\n\n";
  int bytes, rc;
  char *p;
  FILE *fp = NULL, *fp2 = NULL;
  char *buf = NULL;
  char *tp;
  char *ds_start, *ds_end;
  char *cdp_start, *cdp_end;
  char *row_start, *row_end;
  char *v_start, *v_end;
  char *database_start, *database_end;
  long slen;
  int pos;
  int i, j;
  size_t len;
  int err = -1;

  fp = fopen(input_file, "r");
  if(!fp) {
    printf("can't read rrd-file [%s]\n", input_file);
    goto cleanup;
  }
  fp2 = fopen(output_file, "w");
  if(!fp2) {
    printf("can't open rrd-file [%s]\n", output_file);
    goto cleanup;
  }

  rc = fseek(fp, 0, SEEK_END);
  if(rc < 0) {
    printf("fseek error\n");
    goto cleanup;
  }
  slen = ftell(fp);
  if(slen < 0 || fseek(fp, 0, SEEK_SET)<0) {
    printf("fseek error\n");
    goto cleanup;
  }
  buf = (char *)malloc(slen);
  if(!buf) {
    printf("error malloc %ld\n", slen);
    goto cleanup;
  }

  bytes = 0;
  while(!feof(fp)) {
    rc = fread(&buf[bytes], 1, BUFSIZE, fp);
    if(rc <= 0) {
      printf("error read rc=%d\n", rc);
      if(ferror(fp)) {
	printf("file error\n");
	rc = -1;
      }
      if(feof(fp)) {
	//printf("end of file reached\n");
	rc = 0;
      }
      break;
    }
    bytes += rc;
  }
  fclose(fp);
  fp = NULL;

  if(bytes != slen) {
    printf("%d != %ld\n", bytes, slen);
    goto cleanup;
  }
  //printf("%d bytes read\n", bytes);

  p = buf;


  pos = 0;
  tp = p;
  for(i=0; i<column_nr; i++) {
    if(get_section_name(tp, "<ds>", "</ds>", &ds_start, &ds_end) == -1) {
      goto cleanup;
    }
    tp = ds_end;
  }
  tp = &tp[7];  // save carrige return after </ds> too.

  len = (unsigned long)tp - (unsigned long)p;
  rc = fwrite(p, 1, (size_t)len, fp2);
  p = tp;

  if(fprintf(fp2, ds_sect, column_name) < 0) {
    goto cleanup;
  }

  
  for(j=0; j<12; j++) {

    if(get_section_name(tp, "<cdp_prep>", "</cdp_prep>", &cdp_start, &cdp_end) == -1) {
      goto cleanup;
    }
    for(i=0; i<column_nr; i++) {
      if(get_section_name(tp, "<ds>", "</ds>", &ds_start, &ds_end) == -1) {
	goto cleanup;
      }
      tp = ds_end;
    }
    tp = &tp[6];  // save carrige return after </ds> too.

    len = (unsigned long)tp - (unsigned long)p;
    rc = fwrite(p, 1, len, fp2);
    p = tp;

    if(fprintf(fp2, "\t\t\t<ds><value> NaN </value>  <unknown_datapoints> 0 </unknown_datapoints></ds>\n") < 0) {
      goto cleanup;
    }


    if(get_section_name(tp, "<database>", "</database>", &database_start, &database_end) == -1) {
      goto cleanup;
    }
    database_end = &database_end[12];

    while(1) {
      if((get_section_name(tp, "<row>", "</row>", &row_start, &row_end) == -1) || (row_start > database_end)) {
	//printf("no more row sections in file\n");
	if(database_end > p) {
	  len = (unsigned long)database_end - (unsigned long)p;
	  rc = fwrite(p, 1, len, fp2);
	  p = &p[len];
	}
	break;
      }
      tp = &row_start[5];

      for(i=0; i<column_nr; i++) {
	if(get_section_name(tp, "<v>", "</v>", &v_start, &v_end) == -1) {
	  goto cleanup;
	}
	tp = v_end;
      }
      tp = &tp[4];

      len = (unsigned long)tp - (unsigned long)p;
      rc = fwrite(p, 1, len, fp2);
      p = tp;
      if(fprintf(fp2, "<v> NaN </v>") < 0) {
	goto cleanup;
      }
    }
  }

  len = slen - ((unsigned long)database_end - (unsigned long)buf);
  //printf("save rest=%d\n", len);
  rc = fwrite(database_end, 1, len, fp2);


  err = 0 ;
 cleanup:
  if(fp) fclose(fp);
  if(fp2) fclose(fp2);
  if(buf) free(buf);
  return err;
}

int delete_column(int column_nr, char *column_name, char *input_file, char *output_file)
{
  int bytes, rc;
  char *p;
  FILE *fp = NULL, *fp2 = NULL;
  char *buf = NULL;
  char *tp;
  char *ds_start, *ds_end;
  char *name_start, *name_end;
  char *cdp_start, *cdp_end;
  char *row_start, *row_end;
  char *v_start, *v_end;
  char *database_start, *database_end;
  long slen;
  int pos;
  int i, j;
  size_t len = 0;
  int err = -1;
  int nr_columns = 0;
  struct sensor *s;

  s = sensors;
  while(s) {
    nr_columns++;
    s = s->next;
  }
  if(nr_columns == 0) {
    printf("Can't find any sensors in file\n");
    exit(0);
  }
  if(column_nr > nr_columns) {
    printf("Column number is too big. Only found %d columns in file\n", nr_columns);
    exit(0);
  }

  fp = fopen(input_file, "r");
  if(!fp) {
    printf("can't read xml-file [%s]\n", input_file);
    goto cleanup;
  }
  fp2 = fopen(output_file, "w");
  if(!fp2) {
    printf("can't write xml-file [%s]\n", output_file);
    goto cleanup;
  }

  rc = fseek(fp, 0, SEEK_END);
  if(rc < 0) {
    printf("fseek error\n");
    goto cleanup;
  }
  slen = ftell(fp);
  if(slen < 0 || fseek(fp, 0, SEEK_SET)<0) {
    printf("fseek error\n");
    goto cleanup;
  }
  buf = (char *)malloc(slen);
  if(!buf) {
    printf("error malloc %ld\n", slen);
    goto cleanup;
  }

  bytes = 0;
  while(!feof(fp)) {
    rc = fread(&buf[bytes], 1, BUFSIZE, fp);
    if(rc <= 0) {
      printf("error read rc=%d\n", rc);
      if(ferror(fp)) {
	printf("file error\n");
	rc = -1;
      }
      if(feof(fp)) {
	//printf("end of file reached\n");
	rc = 0;
      }
      break;
    }
    bytes += rc;
  }
  fclose(fp);
  fp = NULL;

  if(bytes != slen) {
    printf("%d != %ld\n", bytes, slen);
    goto cleanup;
  }
  //printf("%d bytes read\n", bytes);

  p = buf;


  pos = 0;
  tp = p;
  for(i=0; i<nr_columns; i++) {
    char a[256];
    if(get_section_name(tp, "<ds>", "</ds>", &ds_start, &ds_end) == -1) {
      printf("Error ds");
      goto cleanup;
    }
    if(get_section_name(ds_start, "<name>", "</name>", &name_start, &name_end) == -1) {
      printf("Error name");
      goto cleanup;
    }

    if((column_nr == i) || strstr(column_name, &a[7])) {
      /* remove this */
      column_nr = i;
      //printf("Remove this column %d [%s]\n", i, a);
      if(i == 0) {
	/* Save start if remove first sensor */
	len = (unsigned long)ds_start - (unsigned long)p;
	rc = fwrite(p, 1, (size_t)len, fp2);
      }
      //tp = &ds_end[7]; // save carrige return after </ds> too.
      tp = strchr(&ds_end[1], '<'); // find next tag
    } else {
      //tp = &ds_end[7]; // save carrige return after </ds> too.
      tp = strchr(&ds_end[1], '<'); // find next tag
      
      len = (unsigned long)tp - (unsigned long)p;
      rc = fwrite(p, 1, (size_t)len, fp2);
    }
    p = tp;
  }

  for(j=0; j<12; j++) {
    if(get_section_name(tp, "<cdp_prep>", "</cdp_prep>", &cdp_start, &cdp_end) == -1) {
      printf("Error cdp_prep\n");
      goto cleanup;
    }
    //cdp_end = &cdp_end[12];
    cdp_end = strchr(&cdp_end[1], '<'); // find next tag

    for(i=0; i<nr_columns; i++) {
      if(get_section_name(tp, "<ds>", "</ds>", &ds_start, &ds_end) == -1) {
	printf("Error ds2\n");
	goto cleanup;
      }
      //tp = &ds_end[6]; // save carrige return after </ds> too.
      ds_end = strchr(&ds_end[1], '<'); // find next tag
      if(column_nr == i) {
	/* remove this */
	//printf("Remove this ds row %d\n", i);
	if(i==0) {
	  cdp_start = strchr(&cdp_start[1], '<'); // find next tag
	  len = (unsigned long)cdp_start - (unsigned long)p;
	  rc = fwrite(p, 1, (size_t)len, fp2);
	}
      } else {
	len = (unsigned long)ds_end - (unsigned long)p;
	rc = fwrite(p, 1, (size_t)len, fp2);
      }
      tp = ds_end;
      p = tp;
    }

    if(get_section_name(tp, "<database>", "</database>", &database_start, &database_end) == -1) {
      printf("Error database1\n");
      goto cleanup;
    }
    //database_end = &database_end[12];
    database_end = strchr(&database_end[1], '<');
    database_start = strchr(&database_start[1], '<');

    len = ((unsigned long)database_start - (unsigned long)tp);
    //printf("save rest=%d after cdp_end\n", len);
    rc = fwrite(tp, 1, len, fp2);
    p = tp = database_start;

    while(1) {
      if((get_section_name(tp, "<row>", "</row>", &row_start, &row_end) == -1) || (row_start > database_end)) {
	//printf("no more row sections in file\n");
	if(database_end > p) {
	  len = (unsigned long)database_end - (unsigned long)p;
	  rc = fwrite(p, 1, len, fp2);
	  p = &p[len];
	}
	break;
      }
      tp = &row_start[5];

      for(i=0; i<nr_columns; i++) {
	if(get_section_name(tp, "<v>", "</v>", &v_start, &v_end) == -1) {
	  goto cleanup;
	}
	//tp = &v_end[4];
	v_end = strchr(&v_end[1], '<');
	if(column_nr == i) {
	  /* remove this */
	  //printf("Remove this column %d\n", i);
	  if(i==0) {
	    row_start = strchr(&row_start[1], '<'); // find next tag
	    len = (unsigned long)row_start - (unsigned long)p;
	    rc = fwrite(p, 1, (size_t)len, fp2);
	  }
	} else {
	  len = (unsigned long)v_end - (unsigned long)p;
	  rc = fwrite(p, 1, (size_t)len, fp2);
	}
	tp = v_end;
	p = tp;
      }
      //printf("Remove this column %d\n", i);
    }
  }

  len = slen - ((unsigned long)database_end - (unsigned long)buf);
  printf("save rest=%d\n", len);
  rc = fwrite(database_end, 1, len, fp2);


  err = 0 ;
 cleanup:
  if(fp) fclose(fp);
  if(fp2) fclose(fp2);
  if(buf) free(buf);
  return err;
}

int print_data(char *p, char **np, time_t tm_from, time_t tm_to, int column)
{
  char *s;
  char *buf;
  unsigned long rowtime;
  char *data_start;
  char *data_end;
  char *db_start;
  char *db_end;

  if(!p) return -1;
  if(!np || !*np) return -1;

  if(get_section_name(p, "<rra>", "</rra>",  &data_start, &data_end) == -1) {
    printf("can't find <rra> tags\n");
    return -1;
  }
  if(get_section_name(data_start, "<database>", "</database>",  &db_start, &db_end) == -1) {
    printf("can't find <database> tags\n");
    return -1;
  }

  /* used for next call to change_data */
  *np = db_end;

  s = db_start;
  buf = s;
  while(s < db_end) {
#if 0
    rc = sscanf(s, "%[^\n]\n", buf);
    len = strlen(buf);
#else
    buf = s;
#endif
    //printf("rc=%d  len=%d  buf=[%s]\n", rc, len, buf);
    rowtime = get_time(buf);
    if(rowtime > 0) {
      //printf("found rowtime=%ld\n", rowtime);

      if((rowtime >= (unsigned long)tm_from) &&
	 (rowtime <= (unsigned long)tm_to)) {
	char *a, *b;
	int len1;
	int i;

	//printf("found rowtime=%ld\n", rowtime);
	a = buf;
	for(i=0; i<=column; i++) {
	  a = strstr(a+1, "<v>");
	  if(!a) {
	    printf("error finding %d <v>\n", i);
	    break;
	  }
	}
	if(a) {
	  char tmp[512];
	  int vpos;
	  vpos = a-buf;
	  b = strstr(a, "</v>");
	  if(b) {
	    b += strlen("</v>");
	    len1 = b-a;
	    strncpy(tmp, a, len1);
	    tmp[len1] = '\0';

	    {
	      float f;
	      sscanf(tmp+4, "%e", &f);
	      printf("%ld:%5.1f\n", rowtime, f);
	    }
	  } else {
	    printf("can't find </v>\n");
	  }
    
	}
      }
    }
#if 0
    s += len + 1;  /* carrige return */
#else
    while(*s && *s!='\n') s++;
    s++;
#endif
  }
  return 0;
}

int clear_data(char *p, char **np, time_t tm_from, time_t tm_to, int column)
{
  char *s;
  char *buf;
  unsigned long rowtime;
  char *data_start;
  char *data_end;
  char *db_start;
  char *db_end;
  //struct temp *t;
  //struct temp *oldt;

  if(!p) return -1;
  if(!np || !*np) return -1;

  if(get_section_name(p, "<rra>", "</rra>",  &data_start, &data_end) == -1) {
    printf("can't find <rra> tags\n");
    return -1;
  }
  if(get_section_name(data_start, "<database>", "</database>",  &db_start, &db_end) == -1) {
    printf("can't find <database> tags\n");
    return -1;
  }

  /* used for next call to change_data */
  *np = db_end;

  s = db_start;
  buf = s;
  while(s < db_end) {
#if 0
    rc = sscanf(s, "%[^\n]\n", buf);
    len = strlen(buf);
#else
    buf = s;
#endif
    //printf("rc=%d  len=%d  buf=[%s]\n", rc, len, buf);
    rowtime = get_time(buf);
    if(rowtime > 0) {
      //printf("found rowtime=%ld\n", rowtime);

      if((rowtime >= (unsigned long)tm_from) &&
	 (rowtime <= (unsigned long)tm_to)) {
	char *a, *b;
	int len1, len2;
	int i;

	//printf("found rowtime=%ld\n", rowtime);
	a = buf;
	for(i=0; i<=column; i++) {
	  a = strstr(a+1, "<v>");
	  if(!a) {
	    printf("error finding %d <v>\n", i);
	    break;
	  }
	}
	if(a) {
	  char tmp[512];
	  int vpos;
	  vpos = a-buf;
	  b = strstr(a, "</v>");
	  if(b) {
	    b += strlen("</v>");
	    len1 = b-a;
	    strncpy(tmp, a, len1);
	    tmp[len1] = '\0';

	    len2 = sprintf(tmp, "<v> NaN");

	    while(len2 < len1-4) {
	      tmp[len2++] = ' ';
	    }
	    memcpy(&tmp[len2], "</v>", 4);
	    len2 += 4;
	    tmp[len2] = '\0';
	    /* replace original row */
	    memcpy(&s[vpos], tmp, strlen(tmp));
	  } else {
	    printf("can't find </v>\n");
	  }
    
	}
      }
    }
#if 0
    s += len + 1;  /* carrige return */
#else
    while(*s && *s!='\n') s++;
    s++;
#endif
  }
  return 0;
}


int print_columns()
{
  struct sensor *s = sensors;
  if(!sensors) {
    printf("No sensors found yet\n");
    exit(0);
  }
  while(s) {
    printf("Column %3d [%15s] [%14s]\n", s->column, s->id, s->location);
    s = s->next;
  }
  return 0;
}

int get_location_column(char *location)
{
  struct sensor *s = sensors;
  if(!sensors) {
    printf("No sensors found yet\n");
    exit(0);
  }
  if(!location) return -1;
  while(s) {
    //printf("compare [%s][%s]\n", location, s->location);
    if(!strcmp(location, s->location)) {
      return s->column;
    }
    s = s->next;
  }
  return -1;
}

int get_column_name(int column_nr, char *column_name)
{
  struct sensor *s = sensors;
  if(!sensors) {
    printf("No sensors found yet\n");
    exit(0);
  }
  while(s) {
    if(s->column == column_nr) {
      strcpy(column_name, s->location);
      return 1;
    }
    s = s->next;
  }
  return -1;
}


int get_column_sensid(char *sensid)
{
  struct sensor *s = sensors;
  if(!sensors) {
    printf("No sensors found yet\n");
    exit(0);
  }
  while(s) {
    if(strcmp(s->id, sensid) == 0) {
      return s->column;
    }
    s = s->next;
  }
  return -1;
}


void replace(char *input_file, char *output_file, char *data_file, int column)
{
  int bytes, rc;
  char *p;
  FILE *fp = NULL, *fp2 = NULL;
  char *buf = NULL;
  char *np, *tp;
  long slen;
  int pos;

  fp = fopen(input_file, "r");
  if(!fp) {
    printf("can't read rrd-file [%s]\n", input_file);
    goto cleanup;
  }
  fp2 = fopen(output_file, "w");
  if(!fp2) {
    printf("can't write rrd-file [%s]\n", output_file);
    goto cleanup;
  }

  rc = fseek(fp, 0, SEEK_END);
  if(rc < 0) {
    printf("fseek error\n");
    goto cleanup;
  }
  slen = ftell(fp);
  if(slen < 0 || fseek(fp, 0, SEEK_SET)<0) {
    printf("fseek error\n");
    goto cleanup;
  }
  buf = (char *)malloc(slen);
  if(!buf) {
    printf("error malloc %ld\n", slen);
    goto cleanup;
  }

  bytes = 0;
  while(!feof(fp)) {
    rc = fread(&buf[bytes], 1, BUFSIZE, fp);
    if(rc <= 0) {
      printf("error read rc=%d\n", rc);
      if(ferror(fp)) {
	printf("file error\n");
	rc = -1;
      }
      if(feof(fp)) {
	//printf("end of file reached\n");
	rc = 0;
      }
      break;
    }
    bytes += rc;
  }
  fclose(fp);
  fp = NULL;

  if(bytes != slen) {
    printf("%d != %ld\n", bytes, slen);
    goto cleanup;
  }
  //printf("%d bytes read\n", bytes);

  rc = load_temp(data_file);
  if(rc < 0) {
    goto cleanup;
  }

  p = buf;

  printf("Update AVERAGE 60\n");
  tp = p;
  if(change_data(tp, &np, column) == -1) {
    printf("error update AVERAGE 60\n");
    goto cleanup;
  }
  printf("Update AVERAGE 1800\n");
  tp = np;
  if(change_data(tp, &np, column) == -1) {
    printf("error update AVERAGE 1800\n");
    goto cleanup;
  }
  printf("Update AVERAGE 7200\n");
  tp = np;
  if(change_data(tp, &np, column) == -1) {
    printf("error update AVERAGE 7200\n");
    goto cleanup;
  }
  printf("Update AVERAGE 86400\n");
  tp = np;
  if(change_data(tp, &np, column) == -1) {
    printf("error update AVERAGE 86400\n");
    goto cleanup;
  }
  tp = np;

  pos = 0;
  while(pos < slen) {
    rc = fwrite(&buf[pos], 1, (size_t)((slen-pos > BUFSIZE) ? BUFSIZE : slen-pos), fp2);
    if(rc <= 0) {
      printf("error write rc=%d\n", rc);
      if(ferror(fp2)) {
	printf("file error\n");
	rc = -1;
      }
      break;
    }
    pos += rc;
  }
  fclose(fp2);
  fp2 = NULL;

 cleanup:
  if(fp) fclose(fp);
  if(fp2) fclose(fp2);
  if(buf) free(buf);
}

void debug_print(char *input_file, time_t tm_from, time_t tm_to, int column)
{
  char *p;
  int bytes, rc;
  FILE *fp = NULL;
  char *buf = NULL;
  char *np, *tp;
  long slen;

  fp = fopen(input_file, "r");
  if(!fp) {
    printf("can't read rrd-file [%s]\n", input_file);
    goto cleanup;
  }

  rc = fseek(fp, 0, SEEK_END);
  if(rc < 0) {
    printf("fseek error\n");
    goto cleanup;
  }
  slen = ftell(fp);
  if(slen < 0 || fseek(fp, 0, SEEK_SET)<0) {
    printf("fseek error\n");
    goto cleanup;
  }
  buf = (char *)malloc(slen);
  if(!buf) {
    printf("error malloc %ld\n", slen);
    goto cleanup;
  }

  bytes = 0;
  while(!feof(fp)) {
    rc = fread(&buf[bytes], 1, BUFSIZE, fp);
    if(rc <= 0) {
      printf("error read rc=%d\n", rc);
      if(ferror(fp)) {
	printf("file error\n");
	rc = -1;
      }
      if(feof(fp)) {
	//printf("end of file reached\n");
	rc = 0;
      }
      break;
    }
    bytes += rc;
  }
  fclose(fp);
  fp = NULL;

  if(bytes != slen) {
    printf("%d != %ld\n", bytes, slen);
    goto cleanup;
  }
  //printf("%d bytes read\n", bytes);

  p = buf;

  tp = p;
  printf("Print AVERAGE 60\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print AVERAGE 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Print AVERAGE 1800\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print AVERAGE 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Print AVERAGE 7200\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print AVERAGE 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Print AVERAGE 86400\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print AVERAGE 86400\n");
    goto cleanup;
  }
  tp = np;



  printf("Print MIN 60\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MIN 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MIN 1800\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MIN 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MIN 7200\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MIN 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MIN 86400\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MIN 86400\n");
    goto cleanup;
  }
  tp = np;



  printf("Print MAX 60\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MAX 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MAX 1800\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MAX 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MAX 7200\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MAX 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Print MAX 86400\n");
  if(print_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error print MAX 86400\n");
    goto cleanup;
  }
  tp = np;


 cleanup:
  if(fp) fclose(fp);
  if(buf) free(buf);
}

void zerovalue(char *input_file, char *output_file, time_t tm_from, time_t tm_to, int column)
{
  char *p;
  int bytes, rc;
  FILE *fp = NULL, *fp2 = NULL;
  char *buf = NULL;
  char *np, *tp;
  long slen;
  int pos;

  fp = fopen(input_file, "r");
  if(!fp) {
    printf("can't read rrd-file [%s]\n", input_file);
    goto cleanup;
  }
  fp2 = fopen(output_file, "w");
  if(!fp2) {
    printf("can't write rrd-file [%s]\n", output_file);
    goto cleanup;
  }

  rc = fseek(fp, 0, SEEK_END);
  if(rc < 0) {
    printf("fseek error\n");
    goto cleanup;
  }
  slen = ftell(fp);
  if(slen < 0 || fseek(fp, 0, SEEK_SET)<0) {
    printf("fseek error\n");
    goto cleanup;
  }
  buf = (char *)malloc(slen);
  if(!buf) {
    printf("error malloc %ld\n", slen);
    goto cleanup;
  }

  bytes = 0;
  while(!feof(fp)) {
    rc = fread(&buf[bytes], 1, BUFSIZE, fp);
    if(rc <= 0) {
      printf("error read rc=%d\n", rc);
      if(ferror(fp)) {
	printf("file error\n");
	rc = -1;
      }
      if(feof(fp)) {
	//printf("end of file reached\n");
	rc = 0;
      }
      break;
    }
    bytes += rc;
  }
  fclose(fp);
  fp = NULL;

  if(bytes != slen) {
    printf("%d != %ld\n", bytes, slen);
    goto cleanup;
  }
  //printf("%d bytes read\n", bytes);

  p = buf;

  tp = p;
  printf("Clear AVERAGE 60\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear AVERAGE 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear AVERAGE 1800\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear AVERAGE 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear AVERAGE 7200\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear AVERAGE 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear AVERAGE 86400\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear AVERAGE 86400\n");
    goto cleanup;
  }
  tp = np;



  printf("Clear MIN 60\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MIN 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MIN 1800\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MIN 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MIN 7200\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MIN 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MIN 86400\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MIN 86400\n");
    goto cleanup;
  }
  tp = np;



  printf("Clear MAX 60\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MAX 60\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MAX 1800\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MAX 1800\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MAX 7200\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MAX 7200\n");
    goto cleanup;
  }
  tp = np;
  printf("Clear MAX 86400\n");
  if(clear_data(tp, &np, tm_from, tm_to, column) == -1) {
    printf("error clear MAX 86400\n");
    goto cleanup;
  }
  tp = np;


  pos = 0;
  while(pos < slen) {
    rc = fwrite(&buf[pos], 1, (size_t)((slen-pos > BUFSIZE) ? BUFSIZE : slen-pos), fp2);
    if(rc <= 0) {
      printf("error write rc=%d\n", rc);
      if(ferror(fp2)) {
	printf("file error\n");
	rc = -1;
      }
      break;
    }
    pos += rc;
  }
  fclose(fp2);
  fp2 = NULL;

 cleanup:
  if(fp) fclose(fp);
  if(fp2) fclose(fp2);
  if(buf) free(buf);
}

int main(int argc, char **argv)
{
  char *np;
  int len;
  int n;
  char column_name[80] = "\000";
  int column_nr = -1;
  int mode_print = 0;
  int mode_zero = 0;
  int mode_replace = 0;
  int mode_add = 0;
  int mode_delete = 0;
  time_t tm_from = 0, tm_to = 0;
  char input_file[128] = "\000";
  char output_file[128] = "\000";
  char data_file[128] = "\000";

  n = 1;
  while(n<argc) {
    switch(argv[n][0]) {
    case '-':
      switch(argv[n++][1]) {
      case 'a':
	mode_add = 1;
	break;
      case 'd':
	mode_delete = 1;
	break;
      case 'r':
	mode_replace = 1;
	break;
      case 'z':
	mode_zero = 1;
	break;
      case 'p':
	mode_print = 1;
	break;
      case 'c':
	if(n>=argc) usage(argv);
	//printf("argv[n]=%s len=%d\n", argv[n], strlen(argv[n]));
	if((strlen(argv[n]) == 15) && (argv[n][2]=='.')) {
	  column_nr = get_column_sensid(argv[n]);
	  printf("column_nr=%d argv[n]=%s\n", column_nr, argv[n]);
	  if(column_nr < 0) {
	    //printf("Can't find sensid %s\n", argv[n]);
	    column_nr = -1;
	  } else {
	    get_column_name(column_nr, column_name);
	    //printf("columnname.=[%s]\n", column_name);
	  }
	} else if((strlen(argv[n]) >= 14) && !strchr(argv[n], '.')) {
	  strcpy(column_name, argv[n]);
	  column_nr = get_location_column(argv[n]);
	  if(column_nr < 0) {
	    printf("Can't find column [%s]\n", argv[n]);
	    column_nr = -1;
	  } else {
	    get_column_name(column_nr, column_name);
	    //printf("columnname=[%s] %d\n", column_name, column_nr);
	  }
	} else {
	  column_nr = atoi(argv[n]);
	  if(get_column_name(column_nr, column_name) < 0) {
	    printf("Can't find column nr %d\n", column_nr);
	    column_nr = -1;
	  }
	}
	n++;
	break;
      case 'i':
	if(n>=argc) usage(argv);
	strcpy(input_file, argv[n]);
	if(get_sensors_from_file(input_file) < 0) {
	  printf("Error: Couldn't read input file\n");
	  exit(0);
	}
	n++;
	break;
      case 'o':
	if(n>=argc) usage(argv);
	strcpy(output_file, argv[n]);
	n++;
	break;
      case 'l':
	if(n>=argc) usage(argv);
	strcpy(data_file, argv[n]);
	n++;
	break;
      case 't':
	if(n>=argc) usage(argv);
	np = argv[n];
	len = strlen(np);
	if(np[0] == '-') {
	  tm_from = 0;
	  tm_to = atol(&np[1]);
	} else if(np[len-1] == '-') {
	  tm_from = atol(np);
	  tm_to = (unsigned long)-1;
	} else if(strchr(np, '-')) {
	  tm_from = atol(np);
	  tm_to = atol(strchr(np, '-')+1);
	  if(tm_from > tm_to) {
	    printf("from-time is greater than to-time\n");
	    usage(argv);
	  }
	} else {
	  usage(argv);
	}
	n++;
	break;
      default:
	usage(argv);
	break;
      }
      break;
    default:
      usage(argv);
    }
  }

  if(!input_file[0]) {
    printf("input-file required\n");
    usage(argv);
  }

  if(!mode_replace && !mode_zero && !mode_print && !mode_add && !mode_delete) {
    printf("-r, -z, -p, -a or -d not specified.\n");
    usage(argv);
  }
  if(mode_add) {
    struct sensor *s;
    if(!output_file[0]) {
      printf("no output-file\n");
      usage(argv);
    }
    if(column_nr >= 0) {
      printf("column already exists [%s]\n", column_name);
      usage(argv);
    }
    if(!column_name[0] || (strlen(column_name)>19)) {
      printf("A new column-name have to be specified, eg. -c 10123456789012tempe (length<=19 chars)\n");
      usage(argv);
    }
    column_nr = 0;
    s = sensors;
    while(s) {
      column_nr++;
      s = s->next;
    }
    add_column(column_nr, column_name, input_file, output_file);
  } else if(mode_delete) {
    if(!output_file[0]) {
      printf("no output-file\n");
      usage(argv);
    }
    if(column_nr < 0) {
      printf("couldn't find column %d[%s]\n", column_nr, column_name);
      usage(argv);
    }
    delete_column(column_nr, column_name, input_file, output_file);
  } else if(mode_print) {
    if(!tm_from && !tm_to) {
      tm_to = ((time_t)-1);
    }
    if(column_nr < 0) {
      printf("column not found. Available columns are:\n");
      print_columns();
      usage(argv);
    }
    debug_print(input_file, tm_from, tm_to, column_nr);
    exit(0);
  }
  if(mode_replace) {
    if(!output_file[0]) {
      printf("no output-file\n");
      usage(argv);
    }
    if(!data_file[0]) {
      printf("no data-file\n");
      usage(argv);
    }
    if(column_nr < 0) {
      printf("column not found. Available columns are:\n");
      print_columns();
      usage(argv);
    }
    replace(input_file, output_file, data_file, column_nr);
    exit(0);
  }
  if(mode_zero) {
    if(!output_file[0]) {
      printf("no output-file\n");
      usage(argv);
    }
    if(!tm_from && !tm_to) {
      tm_to = ((time_t)-1);
    }
    if(column_nr < 0) {
      printf("column not found. Available columns are:\n");
      print_columns();
      usage(argv);
    }
    printf("clear column %d [%s]\n", column_nr, column_name);
    zerovalue(input_file, output_file, tm_from, tm_to, column_nr);
  }
  exit(0);
}

int get_sensors_from_file(char *input_file)
{
  int bytes, rc;
  char *p;
  FILE *fp = NULL;
  char *buf = NULL;
  char *tp;
  char sensor_name[30];
  char *ds_start, *ds_end;
  char *name_start, *name_end;
  long slen;
  int pos;
  int err = -1;

  fp = fopen(input_file, "r");
  if(!fp) {
    printf("can't read rrd-file [%s]\n", input_file);
    goto cleanup;
  }

  rc = fseek(fp, 0, SEEK_END);
  if(rc < 0) {
    printf("fseek error\n");
    goto cleanup;
  }
  slen = ftell(fp);
  if(slen < 0 || fseek(fp, 0, SEEK_SET)<0) {
    printf("fseek error\n");
    goto cleanup;
  }
  buf = (char *)malloc(slen);
  if(!buf) {
    printf("error malloc %ld\n", slen);
    goto cleanup;
  }

  bytes = 0;
  while(!feof(fp)) {
    rc = fread(&buf[bytes], 1, BUFSIZE, fp);
    if(rc <= 0) {
      printf("error read rc=%d\n", rc);
      if(ferror(fp)) {
	printf("file error\n");
	rc = -1;
      }
      if(feof(fp)) {
	//printf("end of file reached\n");
	rc = 0;
      }
      break;
    }
    bytes += rc;
  }
  fclose(fp);
  fp = NULL;

  if(bytes != slen) {
    printf("%d != %ld\n", bytes, slen);
    goto cleanup;
  }
  //printf("%d bytes read\n", bytes);

  p = buf;


  pos = 0;
  tp = p;
  while(1) {
    char sens_id[30];
    char data_file[30];
    int i;
    if(get_section_name(tp, "<ds>", "</ds>", &ds_start, &ds_end) == -1) {
      //printf("can't find <ds> tag\n");
      if(sensors) rc = 0;
      break;
    }
    if(get_section_name(ds_start, "<name>", "</name>", &name_start, &name_end) == -1) {
      //printf("can't find <name> tag\n");
      if(sensors) rc = 0;
      break;
    }
    name_start = &name_start[6];
    if(sscanf(name_start, "%s", sensor_name) != 1) {
      goto cleanup;
    }
    sens_id[0] = sensor_name[0];
    sens_id[1] = sensor_name[1];
    sens_id[2] = '.';
    for(i=3; i<15; i++) sens_id[i] = sensor_name[i-1];
    sens_id[i] = 0;
    strcpy(data_file, &sensor_name[i-1]);
    add_sensor(NULL, sens_id, sensor_name, pos, 0, NULL, data_file);
    //printf("Add sensor %d %s\n", pos, sens_id);
    pos++;
    tp = ds_end;
  }  

  err = 0 ;

 cleanup:
  if(fp) fclose(fp);
  if(buf) free(buf);
  return err;
}

