/*
 * $Id: sensor.c,v 1.9 2007/10/12 12:46:48 d1mag Exp $
 *
 * temploggerd -- graph/log utility written for OWFS
 * Written 2005 Christian Magnusson
 * email: mag@mag.cx
 * Released under the GPL
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <globals.h>
#include <rrdfile.h>
#include <sensor.h>
#include <util.h>
#include <simultaneous.h>

struct sensor *sensors = NULL;

struct sensor *add_sensor(char *path, char *id, char *location, int column, int use_rrdfile, char *rrdfile, char *datafile)
{
  struct sensor *s, *ss;
  char file[FILENAME_LEN+1];
  char tmp[FILENAME_LEN+1];
  int new_sensor = 0;
  
  (void) use_rrdfile;
  (void) rrdfile;

  if(!id) {
    //printf("add_sensor: sensor id undefined\n");
    return NULL;
  }

  if(datafile && datafile[0] == '/') {
    //printf("add_sensor: id=[%s] datafile=[%s]\n", id, datafile);
    s = sensors;
    while(s) {
      if(!strcmp(s->id, id)) break;
      s = s->next;
    }
  } else {
    if(datafile) {
      sprintf(file, "%s/%s", id, datafile);
    } else {
      sprintf(file, "%s/", id);
    }
    
    s = sensors;
    while(s) {
      sprintf(tmp, "%s/%s", s->id, s->datafile);
      if(!strcmp(tmp, file)) break;
      s = s->next;
    }
  }
  if(!s) {
    s = (struct sensor *)malloc(sizeof(struct sensor));
    if(!s) {
      printf("Error allocate struct sensor\n");
      return NULL;
    }
    memset(s, 0, sizeof(struct sensor));
    //printf("Create a new sensor object %p\n", s);
    new_sensor = 1;
  }
  strncpy(s->id, id, DEVICE_ID_LEN);
  s->id[DEVICE_ID_LEN] = '\000';
  //printf("Sensor id = [%s]\n", s->id);

  if(location) {
    strncpy(s->location, location, LOCATION_LEN);
    s->location[LOCATION_LEN] = '\000';
    //printf("set location=[%s]\n", s->location);
  }

#ifndef IGNORE_RRDFILE
  if(rrdfile) {
    s->rrdfile = add_rrdfile(rrdfile, NULL, NULL, NULL);
    //printf("set rrdfile=[%s]\n", s->rrdfile->name);
  }
#endif
  if(datafile) {
    strcpy(s->datafile, datafile);
    s->datafile[FILENAME_LEN] = '\000';
    //printf("set datafile=[%s]\n", s->datafile);
  }

  s->column = column;
#ifndef IGNORE_RRDFILE
  if(use_rrdfile) {
    if(!s->rrdfile) {
      s->rrdfile = add_rrdfile(rrdfile_default, NULL, NULL, NULL);
      if(!s->rrdfile) {
	printf("failed setting default rrdfile to %s???\n", s->id);
      } else {
	//printf("Set default rrdfile=[%s] to sensor=[%s]\n", s->rrdfile->name, s->id);
      }
    }
  }
#endif

  if(!s->location[0]) {
    strncpy(s->location, s->id, LOCATION_LEN);
    s->location[LOCATION_LEN] = '\000';
    //printf("set location=[%s]\n", s->location);
  }

  if(s->datafile[0] == '/') {
    strcpy(s->full_path, s->datafile);
    strcpy(s->uncached_path, s->datafile);
  } else {
    if(path) {
      sprintf(s->full_path, "%s/%s/%s", path, s->id, s->datafile);
      sprintf(s->uncached_path, "%s/uncached/%s/%s", path, s->id, s->datafile);
    }
  }
  //printf("set full_path=[%s]\n", s->full_path);

  if(new_sensor) {
    //printf("add_sensor: add [%s/%s]\n", s->id, s->datafile);
    if(!sensors) {
      sensors = s;
    } else {
      ss = sensors;
      while(ss->next) {
	ss = ss->next;
      }
      ss->next = s;
    }
    s->next = NULL;
  } else {
    //printf("add_sensor: modify existing sensor [%s/%s]\n", s->id, s->datafile);
  }
  return s;
}

int change_sensor(char *id, char *datafile, char *location)
{
  char file[FILENAME_LEN+1];
  char tmp[FILENAME_LEN+1];
  struct sensor *s;

  sprintf(file, "%s/%s", id, datafile);

  s = sensors;
  while(s) {
    sprintf(tmp, "%s/%s", s->id, s->datafile);
    if(!strcmp(file, tmp)) {
      //printf("change_sensor: %s to location [%s]\n", s->id, location);
      strncpy(s->location, location, LOCATION_LEN);
      s->location[LOCATION_LEN] = '\000';
      return 1;
    }
    s = s->next;
  }
  return 0;
}

int ignore_sensor(char *id)
{
  struct sensor *s;
  if((s=find_sensor(id))) {
    s->ignored = 1;
    return 1;
  }
  return 0;
}

int free_sensor(struct sensor *s)
{
  struct sensor *n, *ss;
  if(!s) return 0;
  n = sensors;
  ss = NULL;
  while(n) {
    if(n == s) {
      if(!ss) {
	sensors = s->next;
      } else {
	ss->next = s->next;
      }
      //printf("free_sensor: remove [%s]\n", s->id);
      free(s);
      return 1;
    }
    ss = n;
    n = n->next;
  }
  //printf("free_sensor: can't find sensor [%s]\n", s->id);
  return 0;
}

int delete_sensor_by_name(char *id)
{
  return free_sensor(find_sensor(id));
}

int free_sensors(void)
{
  while(sensors) free_sensor(sensors);
  return 0;
}

#define RRDID_LEN 20
// This is the max limit of a data-source name in rrdtools
int rrd_sensid(struct sensor *s, char *rrdid)
{
  char id_[DEVID_LEN+1];
  char datafile_[FILENAME_LEN+1];
  int len;
  if(!s) return -1;
  if(s->datafile[0] == '/') {
    remove_space(s->id, id_);
    len = snprintf(rrdid, RRDID_LEN, "%s", id_);
  } else {
    remove_space(s->id, id_);
    remove_space(s->datafile, datafile_);
    len = snprintf(rrdid, RRDID_LEN, "%s%s", id_, datafile_);
    if(len == RRDID_LEN) {
      /* Make a special name when name is looks like
       * 
       * [2043B90A000000voltA]
       * [2043B90A000000olt2A]  (volt2.A is changed to "olt2A")
       */
      len = snprintf(rrdid, RRDID_LEN, "%s%s", id_, &datafile_[1]);
    }
  }
  rrdid[len] = '\000';
  //printf("rd_sensid RRDID_LEN=%d len=%d id=[%s] s->id=[%s]\n", RRDID_LEN, len, rrdid, s->id);
  return len;
}

int is_tempsensor(unsigned char device_id)
{
  switch(device_id) {
  case 0x10:	/* DS18S20 */
  case 0x28:	/* DS18B20 */
  case 0x22:	/* DS1822  */
  case 0x21:	/* DS1921  */
  case 0x1B:	/* DS2436  */
  case 0x26:	/* DS2438  */
  case 0x30:	/* DS2760  */
    return 1;
  default:
    return 0;
  }
}

unsigned char sensor_id(struct sensor *s)
{
  if(s == NULL) {
    printf("sensor_id: s==NULL\n");
    return 0;
  }
  return 16*char2hex(s->id[0]) + char2hex(s->id[1]);
}

struct sensor *find_sensor_from(struct sensor *s, char *id)
{
  if(!s) return NULL;
  while(s) {
    if(!strcmp(id, s->id)) {
      break;
    }
    s = s->next;
  }
  //if(s) printf("found sensor [%s]\n", s->id);
  return s;
}

struct sensor *find_sensor(char *id)
{
  return find_sensor_from(sensors, id);
}

struct sensor *find_sensor_and_datafile(char *id, char *datafile)
{
  struct sensor *ss = sensors;
  struct sensor *s;
  while(ss && (s = find_sensor_from(ss, id))) {
    if(!strcmp(s->datafile, datafile)) return s;
    ss = s->next;
  }
  return NULL;
}

int nr_avail_sensors(void)
{
  struct sensor *s = sensors;
  int nr_avail = 0;
  while(s) {
    if(s->avail) {
      nr_avail++;
    }
    s = s->next;
  }
  return nr_avail;
}

void update_sensors(char *path)
{
  struct sensor *s = sensors;
  float f;
  FILE *fp = NULL;
  int rc;
#ifndef IGNORE_SIMULTANEOUS
  char simul_path[FILENAME_LEN+1];
  struct simultaneous *sim = simultaneous_files;
  while(sim) {
    /* start conversion on all sensors */
    snprintf(simul_path, FILENAME_LEN, "%s%s", path, sim->path);
    if((fp = fopen(simul_path, "w")) != NULL) {
      fwrite("1", 1, 1, fp);
      fclose(fp);
    }
    sim = sim->next;
  }
#else
  (void) path;
#endif
  while(s) {
    s->avail = 0;
    //printf("update_sensors: open [%s]\n", s->full_path);
    if(s->ignored || !s->file_found_on_bus) {
      //printf("update_sensor: ignore reading [%s] ffob=%d ignored=%d [%s]\n", s->id, s->file_found_on_bus, s->ignored, s->full_path);
      s = s->next;
      continue;
    }
//#define READ_UNCACHED 1
#ifdef READ_UNCACHED
   if((fp = fopen(s->uncached_path, "r")) == NULL) {
#else
   if((fp = fopen(s->full_path, "r")) == NULL) {
#endif
      if(s->file_found_on_bus) {
	/* Just warn if it existed on the bus last time */
	printf("update_sensors: Error open sensor %s path=(%s)\n", s->id, s->full_path);
      }
      s->file_found_on_bus = 0;
      s = s->next;
      continue;
    }
    rc = fscanf(fp, "%f", &f);
    if(rc < 0) {
      printf("update_sensors: Error reading %s (%s)\n", s->full_path, strerror(errno));
      s->file_found_on_bus = 0;
    } else if(rc == 1) {
      s->last_value = f;
      s->avail = 1;
      s->file_found_on_bus = 1;
      //printf("update [%s] = %5.2f\n", s->full_path, f);
    } else {
      printf("update_sensors: Empty file %s\n", s->full_path);
    }
    fclose(fp);
    s = s->next;
  }
  return;
}

int print_sensor(struct sensor *s)
{
#if 0
  if(!s) {
    printf("s == NULL\n");
    exit(0);
  }
  printf("id=[%s] datafile=[%s] rrdfile=[%s]\n  full_path=[%s]\n",
	 s->id, s->datafile, s->rrdfile?s->rrdfile->name:"(none)", s->full_path);
  printf("  col=%d avail=%d config=%d ignore=%d d_on_bus=%d f_on_bus=%d\n",
	 s->column, s->avail, s->config_sensor, s->ignored, s->device_found_on_bus,
	 s->file_found_on_bus);
#else
  (void) s;
#endif
  return 0; 
}
