#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <dirent.h>
#include <errno.h>
#include <time.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <ctype.h>
#include <stdarg.h>

#include "sensor.h"
#include "scan_bus.h"
#include "util.h"

#define LOOP_DELAY 100

int dummy_cnt = 0;
int print_results = 0;
int read_buttons = 0;

//#define MAX_ERRORS 5
//#define FILENAME_LEN 80

char rrdfile_default[FILENAME_LEN+1];

#define ABS(x) ((x) < 0 ? -(x) : (x))

enum mess_t { STATIC_MESSAGE=0, TIMED_MESSAGE };

enum swname {
  BUTTON0=0,
  BUTTON1,
  BUTTON2,
  BUTTON3,
  BUTTON4,
  BUTTON5,
  BUTTON6,
  BUTTON7,
};

extern int lcd0_pressed(char *fil, int nr);
extern int lcd1_pressed(char *fil, int nr);
extern int sw_pressed(char *fil, int nr);
extern int lcd0_released(char *fil, int nr);
extern int lcd1_released(char *fil, int nr);
extern int sw_released(char *fil, int nr);

#define ADDR_LCD0 "FF.710200000100"
#define ADDR_LCD1 "FF.720200000100"
#define ADDR_OUTDOOR "10.6D0150000800"

#define ACTIVE_LCD "FF.710200000100"

#define ADDR_SW0 "12.24F824000000"
#define ADDR_SW1 "12.5E1A24000000"
#define ADDR_SW2 "29.000000000000"

#define LCD_ROWLEN 16
#define LCD_ROWS 2
#define LCD_BACKLIGHT_TIMEOUT 20


#define FLOAT_LEN 12
#define FOUR_FLOAT_LEN (4*FLOAT_LEN+3)

char device_path[80];
char uncached_device_path[80];

#define DEVID_LEN 15
struct mess {
  char txt[LCD_ROWLEN+1];
  char devid[DEVID_LEN+1];
  int row;
  unsigned long timeout_ms;
  struct timeval start_time;
  struct timeval end_time;
  struct mess *next;
};

struct mess *lcdmessages = NULL;
int add_message(char *txt, char *id, int row, unsigned long timeout_ms);
struct mess *get_message(char *id, int row, enum mess_t timed);
int delete_message(struct mess *mm);

struct lcd_display {
  char name[16];
  struct timeval last_button;
  int backlight;
  int poweron;
} lcd_displays[] = {
  { ADDR_LCD0, {0,0,}, 0, 0 },
  { ADDR_LCD1, {0,0,}, 0, 0 },
  { "", {0,0,}, 0, 0 },
};

struct remote_button {
  char function[20];
  char on_switch_name[16];
  int  on_switch_port;
  char off_switch_name[16];
  int  off_switch_port;
  int on;
} remote_buttons[] = {
  { "Fan", ADDR_SW2, BUTTON0, ADDR_SW2, BUTTON1, 0 },
  { "LampOffice", ADDR_SW2, BUTTON2, ADDR_SW2, BUTTON3, 0 },
  { "LampSauna", ADDR_SW2, BUTTON4, ADDR_SW2, BUTTON5, 0 },
  { "", "", 0, "", 0, 0 },
};


struct button {
  char name[16];
  int button;
  int last_val;
  int last_pressed;
  int reset;
  int (*pressed)(char *, int);
  int (*released)(char *, int);
} buttons[] = {
  { ADDR_LCD0, BUTTON0, 0, 0, 0, lcd0_pressed, lcd0_released },
  { ADDR_LCD0, BUTTON1, 0, 0, 0, lcd0_pressed, lcd0_released },
  { ADDR_LCD0, BUTTON2, 0, 0, 0, lcd0_pressed, lcd0_released },
  { ADDR_LCD0, BUTTON3, 0, 0, 0, lcd0_pressed, lcd0_released },
  { ADDR_LCD1, BUTTON0, 0, 0, 0, lcd1_pressed, lcd1_released },
  { ADDR_LCD1, BUTTON1, 0, 0, 0, lcd1_pressed, lcd1_released },
  { ADDR_LCD1, BUTTON2, 0, 0, 0, lcd1_pressed, lcd1_released },
  { ADDR_LCD1, BUTTON3, 0, 0, 0, lcd1_pressed, lcd1_released },
  { ADDR_SW0,  BUTTON0, 0, 0, 0, sw_pressed, NULL },
  { ADDR_SW0,  BUTTON1, 0, 0, 0, sw_pressed, NULL },
  { ADDR_SW1,  BUTTON0, 0, 0, 0, sw_pressed, NULL },
  { ADDR_SW1,  BUTTON1, 0, 0, 0, sw_pressed, NULL },
  { "", 0, 0, 0, 0, NULL, NULL },
};

void timeval_add_ms(struct timeval *res, struct timeval *tv, unsigned long ms)
{
  //printf("timeval_add_ms()\n");
  if(!res) return;
  res->tv_sec = tv->tv_sec;
  res->tv_usec = tv->tv_usec;
  
  res->tv_usec += ms*1000;
  while(res->tv_usec >= 1000000) {
    res->tv_usec -= 1000000;
    res->tv_sec++;
  }
  return;
}

void timeval_add(struct timeval *res, struct timeval *tv1, struct timeval *tv2)
{
  //printf("timeval_add()\n");
  if(!res) return;
  res->tv_sec = tv1->tv_sec + tv2->tv_sec;
  res->tv_usec = tv1->tv_usec + tv2->tv_usec;
  
  while(res->tv_usec >= 1000000) {
    res->tv_usec -= 1000000;
    res->tv_sec++;
  }
  return;
}

int timeval_diff(struct timeval *diff,
		 struct timeval *tv1, struct timeval *tv2)
{
  /*
   * tv1 should be greater or equal to tv2
   *    returns 1 if tv1 > tv2
   *    returns 0 if tv1 == tv2
   *
   *    returns -1 if tv1 < tv2
   */
  int res = 0;
  //printf("timeval_diff()\n");

  if(tv1->tv_sec > tv2->tv_sec) res = 1;
  else if(tv1->tv_sec < tv2->tv_sec) res = -1;
  else {
    if(tv1->tv_usec > tv2->tv_usec) res = 1;
    else if(tv1->tv_usec < tv2->tv_usec) res = -1;
    else res = 0;
  }
  if(diff == NULL) return res;

  if(res == 0) {
    diff->tv_sec = 0;
    diff->tv_usec = 0;
  } else if(res == 1) {
    diff->tv_sec = tv1->tv_sec - tv2->tv_sec;
    diff->tv_usec = tv1->tv_usec - tv2->tv_usec;
    if(diff->tv_usec < 0) {
      diff->tv_usec += 1000000;
      diff->tv_sec--;
    }
  } else if(res == -1) {
    diff->tv_sec = tv2->tv_sec - tv1->tv_sec;
    diff->tv_usec = tv2->tv_usec - tv1->tv_usec;
    if(diff->tv_usec < 0) {
      diff->tv_usec += 1000000;
      diff->tv_sec--;
    }
  }
  return res;
}

unsigned long timeval2ms(struct timeval *tv)
{
  unsigned long d;
  d = (tv->tv_sec & 0x003FFFFF) * 1000 + tv->tv_usec / 1000;
  /*
    3FFFFFh == 4194303,
    1000*4194303 == 4194303000 == 48.5 days difference max
  */
  return d;
}

#if 0
void my_delay(unsigned long ms)
{
  struct timespec ts;
  struct timespec rem;

  rem.tv_sec = ms / 1000;
  rem.tv_nsec = 1000000*(ms%1000);
  while(1) {
    ts.tv_sec = rem.tv_sec;
    ts.tv_nsec = rem.tv_nsec;
    if(nanosleep(&ts, &rem) < 0) {
      if(errno == EINTR) {
	/* was interupted... continue sleeping... */
      } else {
	printf("my_delay: error? %s\n", strerror(errno));
	return;
      }
    } else {
      /* completed sleeping */
      return;
    }
  }
}
#endif

int print_row(char *id, int row, char *fmt, ...)
{
  char tmp[LCD_ROWLEN+2];
  char filename[FILENAME_LEN+1];
  struct sensor *s;
  FILE *fp;
  int sz;
  va_list va;
  va_start(va, fmt);

  if(!(s = find_sensor(id))) {
    printf("print_row: can't find sensor [%s]\n", id);
    return -1;
  }
  
  sz = vsnprintf(tmp, LCD_ROWLEN+1, fmt, va);
  while(sz<LCD_ROWLEN) tmp[sz++] = ' ';
  tmp[LCD_ROWLEN] = '\000';

  //printf("print_row: called [%s]\n", tmp);

  sprintf(filename, "%sline16.%d", s->uncached_path, row);
  fp = fopen(filename, "w");
  if(!fp) {
    printf("print_row: error open file [%s]\n", filename);
    va_end(va);
    return -1;
  }
#ifndef __UCLIBC__
  printf("[%s] to [%s]\n", tmp, filename);
#endif
  sz = fwrite(tmp, LCD_ROWLEN, 1, fp);
  fclose(fp);
  if(sz <= 0) {
    printf("print_row: fwrite(): %s %s\n", filename, strerror(errno));
    va_end(va);
    return -1;
  }
  //printf("print_row: done\n");
  va_end(va);
  return 0;
}

struct lcd_display *find_lcd(char *name)
{
  int i = 0;
  while(lcd_displays[i].name[0]) {
    if(strcmp(lcd_displays[i].name, name) == 0) return &lcd_displays[i];
    i++;
  }
  return NULL;
}

struct timeval *lcd_get_last_button(char *name)
{
  struct lcd_display *lcd = find_lcd(name);
  if(!lcd) return NULL;
  return &lcd->last_button;
}

int lcd_set_last_button(char *name)
{
  struct lcd_display *lcd = find_lcd(name);
  if(!lcd) return -1;
  if(gettimeofday(&lcd->last_button, NULL) < 0) {
    printf("lcd_set_last_button: gettimeofday %s\n", strerror(errno));
  }
  return 0;
}

int lcd_get_poweron(char *name)
{
  struct lcd_display *lcd = find_lcd(name);
  if(!lcd) return -1;
  return lcd->poweron;
}

int lcd_set_poweron(char *name, int on)
{
  struct lcd_display *lcd = find_lcd(name);
  struct sensor *s;
  FILE *fp;
  int sz;
  char tmp[2];
  char filename[FILENAME_LEN+1];

  if(!(s=find_sensor(name))) {
    printf("lcd_set_poweron: can't find [%s]\n", name);
    return -1;
  }

  sprintf(filename, "%sLCDon", s->uncached_path);
  fp = fopen(filename, "w");
  if(!fp) {
    printf("lcd_set_poweron: fopen(): %s: %s\n", filename, strerror(errno));
    return -1;
  }
  tmp[0] = (on ? '1':'0');
  tmp[1] = '\000';
  sz = fwrite(tmp, 1, 1, fp);
  fclose(fp);
  if(sz <= 0) {
    printf("lcd_set_poweron: fwrite(): %s %s\n", filename, strerror(errno));
    return -1;
  }
  printf("poweron set to %d\n", on);
  if(lcd) lcd->poweron = on;
  return 0;
}

struct remote_button *find_remote_button(char *function)
{
  int i = 0;
  while(*remote_buttons[i].function) {
    if(!strcmp(remote_buttons[i].function, function)) return &remote_buttons[i];
    i++;
  }
  return NULL;
}

int remote_button_write(char *name, int on)
{
  FILE *fp;
  struct remote_button *rb;
  int sz;
  int i;
  char tmp[2];
  char filename[FILENAME_LEN+1];
  struct sensor *s;

  if(!(s=find_sensor(name))) {
    printf("remote_button_write: can't find [%s]\n", name);
    return -1;
  }

  rb = find_remote_button(name);
  if(!rb) {
    printf("Can't find button\n");
    return -1;
  }

  for(i=0; i<2; i++) {
    if(on) {
      sprintf(filename, "%s%s/PIO.%c", s->uncached_path,
	      rb->on_switch_name, 'A'+rb->on_switch_port);
    } else {
      sprintf(filename, "%s%s/PIO.%c", s->uncached_path,
	      rb->off_switch_name, 'A'+rb->off_switch_port);
    }
    fp = fopen(filename, "w");
    if(!fp) {
      printf("switch_write: fopen(): %s: %s\n", filename, strerror(errno));
      return -1;
    }
    tmp[0] = ((i==0) ? '1':'0');
    tmp[1] = '\000';
    sz = fwrite(tmp, 1, 1, fp);
    fclose(fp);
    if(sz <= 0) {
      printf("remote_button_write: fwrite(): %s %s\n", filename, strerror(errno));
      return -1;
    }
    my_delay(500);
  }
  rb->on = on;
  return 0;
}

int switch_write(char *name, int on)
{
  FILE *fp;
  int sz;
  char tmp[2];
  char filename[FILENAME_LEN+1];
  struct sensor *s;

  if(!(s=find_sensor(name))) {
    printf("switch_write: can't find [%s]\n", name);
    return -1;
  }

  sprintf(filename, "%sPIO.A", s->uncached_path);
  fp = fopen(filename, "w");
  if(!fp) {
    printf("switch_write: fopen(): %s: %s\n", filename, strerror(errno));
    return -1;
  }
  tmp[0] = (on ? '1':'0');
  tmp[1] = '\000';
  sz = fwrite(tmp, 1, 1, fp);
  fclose(fp);
  if(sz <= 0) {
    printf("switch_write: fwrite(): %s %s\n", filename, strerror(errno));
    return -1;
  }
  return 0;
}

int lcd_get_backlight(char *name)
{
  struct lcd_display *lcd = find_lcd(name);
  if(!lcd) return -1;
  return lcd->backlight;
}


int lcd_set_backlight(char *name, int on)
{
  struct lcd_display *lcd = find_lcd(name);
  FILE *fp;
  int sz;
  char tmp[2];
  char filename[FILENAME_LEN+1];
  struct sensor *s;

  if(!(s=find_sensor(name))) {
    printf("lcd_set_backlight: can't find [%s]\n", name);
    return -1;
  }

  sprintf(filename, "%sbacklight", s->uncached_path);
  fp = fopen(filename, "w");
  if(!fp) {
    printf("lcd_set_backlight: fopen(): %s: %s\n", filename, strerror(errno));
    return -1;
  }
  tmp[0] = (on ? '1':'0');
  tmp[1] = '\000';
  sz = fwrite(tmp, 1, 1, fp);
  fclose(fp);
  if(sz <= 0) {
    printf("lcd_set_backlight: fwrite(): %s %s\n", filename, strerror(errno));
    return -1;
  }
  printf("[%s] backlight %s\n", s->id, (on?"on":"off"));
  if(lcd) lcd->backlight = on;
  return 0;
}

int print_time()
{
  char tmp[LCD_ROWLEN+1];
  int len;
  struct tm *tm1;
  time_t t;

  t = time(NULL);
  tm1 = localtime(&t);
  if(tm1 == NULL) {
    printf("print_time: localtime failed\n");
    return -1;
  }
  len = strftime(tmp, LCD_ROWLEN, "%d/%m %H:%M:%S", tm1);
  if(len <= 0) {
    printf("print_time: strftime failed len=%d\n", len);
    return -1;
  }
  add_message(tmp, ACTIVE_LCD, 1, 0);
  return 0;
}

int print_temp()
{
  FILE *fp;
  char filename[FILENAME_LEN+1];
  char tmp[FLOAT_LEN+1];
  int sz;
  float t;
  struct sensor *s;

  //printf("print_temp: called\n");
#ifdef __UCLIBC__
  sprintf(filename, "/var/httpd/%s.txt", ADDR_OUTDOOR);
#else
  s = find_sensor(ADDR_OUTDOOR);
  if(!s) {
    return -1;
  }
  sprintf(filename, "%s", s->uncached_path);
#endif
  fp = fopen(filename, "r");
  if(!fp) {
    printf("print_temp: can't open file %s (%s)\n", filename, strerror(errno));
    return -1;
  }
  sz = fread(tmp, FLOAT_LEN, 1, fp);
  fclose(fp);
  if(sz <= 0) {
    printf("print_temp: failed reading %s sz=%d (%s)\n", filename, sz, strerror(errno));
    return -1;
  }
  tmp[FLOAT_LEN] = '\000';

  if(sscanf(tmp, "%f", &t) == 1) {
    sprintf(tmp, "Utomhus: %5.2f", t);
    add_message(tmp, ACTIVE_LCD, 1, 0);
  }
  return 0;
}

int lcd_reset_button(char *name)
{
  char tmp[FOUR_FLOAT_LEN+1];
  char filename[FILENAME_LEN+1];
  FILE *fp;
  int sz;
  struct sensor *s;

  if(!(s=find_sensor(name))) {
    printf("lcd_reset_button: can't find [%s]\n", name);
    return -1;
  }
  sprintf(filename, "%scumulative.ALL", s->uncached_path);
  fp = fopen(filename, "r");
  if(!fp) {
    printf("lcd_reset_button: fopen(): %s: %s\n", filename, strerror(errno));
    return -1;
  }
  sz = fread(tmp, FOUR_FLOAT_LEN, 1, fp);
  fclose(fp);
  if(sz <= 0) {
    printf("lcd_reset_button: fread(): %s %s\n", filename, strerror(errno));
    return -1;
  }
  return 0;
}

int try_reset(char *name)
{
  int i = 0;
  int first_reset = 1;
  while(*buttons[i].name) {
    if(!strcmp(buttons[i].name, name) && !buttons[i].reset) {
      //printf("(not reset %d)\n", i);
      if(first_reset) {
	//printf("do reset [%s]\n", name);
	lcd_reset_button(name);
	first_reset = 0;
      }
      buttons[i].reset = 1;
    }
    i++;
  }
  return 0;
}

int handle_button(char *name, int butt_nr,
		  int nr_pressed, int curval)
{
  int i = 0;
  int found = 0;
  printf("handle_button: [%s] bnr=%d nrpres=%d val=%d\n", name, butt_nr, nr_pressed, curval);
  while(*buttons[i].name) {
    if(!strcmp(buttons[i].name, name)) {
      found = 1;
      break;
    }
    i++;
  }
  if(!found) {
    printf("handle_button: can't find button in list\n");
    return 0;
  }
  //printf("handle_button: found at %d\n", i);
  if(curval < 0) {
    if(nr_pressed & 1)
      curval = !buttons[i].last_val;
    else
      curval = buttons[i].last_val;
  }
  if((buttons[i].last_val == curval) && (nr_pressed & 1)) {
    printf("handle_button: try to fix nr_pressed (nr_pressed=%d)\n", nr_pressed);
    nr_pressed++;
  }

  if(nr_pressed < 0 || nr_pressed > 20) nr_pressed = 0;
  while(nr_pressed>0) {
    //printf("nr_pressed = %d\n", nr_pressed);
    if(!(nr_pressed & 1)) {
      if(curval) {
	if(buttons[i].released) {
	  (buttons[i].released)(name, butt_nr);
	}
      } else {
	if(buttons[i].pressed) {
	  (buttons[i].pressed)(name, butt_nr);
	}
      }
    } else {
      if(curval) {
	if(buttons[i].pressed) {
	  (buttons[i].pressed)(name, butt_nr);
	}
      } else {
	if(buttons[i].released) {
	  (buttons[i].released)(name, butt_nr);
	}
      }
    }
    nr_pressed--;
  }
  buttons[i].last_val = curval;
  return 0;
}

int read_button(struct sensor *s)
{
  char filename[FILENAME_LEN+1];
  char tmp[FOUR_FLOAT_LEN+1];
  FILE *fp;
  int sz, val;
  int but[4];
  int rc, i;
  char *name = s->id;
  if(!*device_path) {
    /* fuse-dev probably not found */
    return -1;
  }
  

  if(sensor_id(s) == 0x12) {
    sprintf(filename, "%slatch.A", s->uncached_path);
    fp = fopen(filename, "r");
    if(!fp) {
      //printf("read_button: fopen(): %s: %s\n", filename, strerror(errno));
      return -errno;
    }
    *tmp = '\0';
    sz = fread(tmp, 1, 1, fp);
    fclose(fp);
    if(sz <= 0) {
      printf("read_button: fread(): %s %s\n", filename, strerror(errno));
      return -errno;
    }
    tmp[sz] = '\0';
    
    if(*tmp == '1') {
      // value has changed according to latch register
      sprintf(filename, "%ssensed.A", s->uncached_path);
      fp = fopen(filename, "r");
      if(!fp) {
	printf("read_button: fopen(): %s: %s\n", filename, strerror(errno));
	return -errno;
      }
      *tmp = '\0';
      sz = fread(tmp, 1, 1, fp);
      fclose(fp);
      if(sz <= 0) {
	printf("read_button: fread(): %s %s\n", filename, strerror(errno));
	return -errno;
      }
      tmp[sz] = '\0';
      val = !atoi(tmp);
      handle_button(name, BUTTON0, 2, val);
    } else {
      // value has NOT changed according to latch register
      sprintf(filename, "%ssensed.A", s->uncached_path);
      fp = fopen(filename, "r");
      if(!fp) {
	printf("read_button: fopen(): %s: %s\n", filename, strerror(errno));
	return -errno;
      }
      *tmp = '\0';
      sz = fread(tmp, 1, 1, fp);
      fclose(fp);
      if(sz <= 0) {
	printf("read_button: fread(): %s %s\n", filename, strerror(errno));
	return -errno;
      }
      tmp[sz] = '\0';
    }
  } else if(sensor_id(s) == 0xFF) {
    try_reset(name);

    *tmp = '\0';
    sprintf(filename, "%scounters.ALL", s->uncached_path);
    fp = fopen(filename, "r");
    if(!fp) {
      printf("read_button: fopen(): %s: %s\n", filename, strerror(errno));
      return -errno;
    }
    sz = fread(tmp, 1, FOUR_FLOAT_LEN, fp);
    fclose(fp);
    if(sz <= 0) {
      printf("read_button: fread(): sz=%d %s %s\n", sz, filename, strerror(errno));
      return -errno;
    }
    tmp[sz] = '\0';

#if 1
    if(print_results) {
      //printf("reading: [%s]\n", tmp);
      printf("reading %d: [%s]\n", ++dummy_cnt, tmp);
    }
#endif
    for(i=0; i<4; i++) but[i] = 0;
    rc = sscanf(tmp, "%d,%d,%d,%d", &but[0], &but[1], &but[2], &but[3]);
    if(rc < 4) {
      printf("read_button: couldn't read 4 values [%s]\n", tmp);
      return -1;
    }
    for(i=0; i<4; i++) {
      if(but[i]) {
	if(!lcd_get_backlight(name)) {
	  lcd_set_backlight(name, 1);
	}
	lcd_set_last_button(name);
	handle_button(name, BUTTON0 + i, but[i], -1);
     }
    }
  } else {
    printf("Can't handle button on that device [%s]\n", name);
  }
  return 1;
}

int lcd0_pressed(char *name, int nr)
{
#if 0
  char tmp[20];
  sprintf(tmp, "lcd0 pressed %d", nr);
  add_message(tmp, name, 0, 1000);
#endif
  lcd1_pressed(name, nr);
  return 0;
}

int lcd0_released(char *name, int nr)
{
#if 0
  char tmp[20];
  sprintf(tmp, "lcd0 released %d", nr);
  add_message(tmp, name, 0, 1000);
  //printf("lcd0_released %d\n", nr);
#endif
  lcd1_released(name, nr);
  return 0;
}

int fan_status = 0;
int lamp_status = 0;

int lcd1_pressed(char *name, int nr)
{
#if 0
  char tmp[20];
  sprintf(tmp, "Button %d", nr);
  add_message(tmp, name, 0, 1000);
#endif
  //printf("lcd1_pressed button%d\n", nr);

  if(nr==0) {
    lamp_status = !lamp_status;
    if(lamp_status) {
      add_message("Turn on lamp", name, 0, 1000);
      switch_write(ADDR_SW0, 1);
      my_delay(500);
      switch_write(ADDR_SW0, 0);
    } else {
      add_message("Turn off lamp", name, 0, 1000);
      switch_write(ADDR_SW1, 1);
      my_delay(500);
      switch_write(ADDR_SW1, 0);
    }
  } else if(nr==1) {

  } else if(nr==2) {
    fan_status = !fan_status;
    if(fan_status) {
      add_message("Turn on fan", name, 0, 1000);
    } else {
      add_message("Turn off fan", name, 0, 1000);
    }
    remote_button_write("Fan", fan_status);
  } else if(nr==3) {
    //add_message("I love Jelena", name, 0, 0);
    add_message("Button pressed", name, 0, 1000);
  } else {
    printf("unknown button %d\n", nr);
  }
  return 0;
}

int lcd1_released(char *name, int nr)
{
#if 0
  char tmp[20];
  sprintf(tmp, "lcd1 released %d", nr);
  add_message(tmp, name, 0, 3000);
#endif
  //printf("lcd1_released %d\n", nr);
  return 0;
}

int sw_pressed(char *name, int nr)
{
  printf("sw_pressed %d\n", nr);
  return 0;
}

int sw_released(char *name, int nr)
{
  printf("sw_released %d\n", nr);
  return 0;
}

int print_message(struct mess *m)
{
  if(!m) return -1;
  if(gettimeofday(&m->start_time, NULL) < 0) {
    printf("print_message: gettimeofday %s\n", strerror(errno));
  }
  timeval_add_ms(&m->end_time, &m->start_time, m->timeout_ms);
  print_row( m->devid, m->row, m->txt);
  return 0;
}

int add_message(char *txt, char *id, int row, unsigned long timeout_ms)
{
  struct mess *m;
  struct mess *newm;
  int timed_message_exists = 0;

  //printf("add_message: [%s]\n", txt);

  newm = malloc(sizeof(struct mess));
  if(!newm) return -1;
  memset(newm, 0, sizeof(struct mess));
  memcpy(newm->txt, txt, LCD_ROWLEN);
  newm->txt[LCD_ROWLEN] = '\000';

  strncpy(newm->devid, id, DEVID_LEN);
  newm->devid[DEVID_LEN] = '\000';
  newm->row = row;
  newm->timeout_ms = timeout_ms;
  if(gettimeofday(&newm->start_time, NULL) < 0) {
    printf("print_message: add_message %s\n", strerror(errno));
  }

  timeval_add_ms(&newm->end_time, &newm->start_time, timeout_ms);

  m = get_message(id, row, TIMED_MESSAGE);
  if(m) {
    timed_message_exists = 1;
  }

  if(timeout_ms == 0) {
    // static message
    m = get_message(id, row, STATIC_MESSAGE);
    if(m) {
      newm->next = m->next;
      memcpy(m, newm, sizeof(struct mess)); // overwrite existing message
      free(newm);
      newm = m;
    } else {
      newm->next = lcdmessages;
      lcdmessages = newm;
    }
  } else {
    // timed message
    if(lcdmessages) {
      m = lcdmessages;
      while(m->next) {
	m = m->next;
      }
      m->next = newm;
    } else {
      lcdmessages = newm;
    }
  }
  if(!timed_message_exists) {
    print_message(newm);
  }
  return 0;
}

struct mess *get_message(char *id, int row, enum mess_t timed)
{
  struct mess *m = lcdmessages;
  if(!m) return NULL;
  while(m) {
    if((timed==TIMED_MESSAGE) && m->timeout_ms &&
       (m->row == row) && !strcmp(m->devid, id)) {
      //printf("Found timed message [%s] row=%d expire=%d now=%d\n", m->txt, m->row, m->end_time, time(NULL));
      // times message
      break;
    }
    if((timed==STATIC_MESSAGE) && !m->timeout_ms &&
       (m->row == row) && !strcmp(m->devid, id)) {
      // static message
      //printf("Found static message [%s] row=%d expire=%d now=%d\n", m->txt, m->row, m->end_time, time(NULL));
      break;
    }
    m = m->next;
  }
  return m;
}

int delete_message(struct mess *mm)
{
  struct mess *m = lcdmessages;
  struct mess *lastm = NULL;
  if(!lcdmessages) {
    printf("delete_message: NO MESSAGES in list!\n");
    return -1;
  }
  while(m) {
    if(m == mm) {
      if(lastm) {
	lastm->next = m->next;
      } else {
	lcdmessages = m->next;
      }
      free(mm);
      //printf("delete_message: message deleted\n");
      return 0;
    }
    lastm = m;
    m = m->next;
  }
  printf("delete_message: not found!\n");
  return -1;
}

void usage(char **argv)
{
  printf("Usage: %s [-d loop_delay] [-p]\n", argv[0]);
  printf("  -d\tLoop delay (default %dms) between polling buttons\n", LOOP_DELAY);
  printf("  -b\tRead buttons on LCDs\n");
  printf("  -p\tPrint all results\n");
  exit(0);
}


int main(int argc, char **argv)
{
  int i, rc;
  struct timeval tvnow;
  struct timeval last_update;
  int nr_errors, nr_sensors;
  unsigned char found_fuse = 0;
  unsigned char time_printed = 0;
  unsigned char first_loop = 1;
  int arg;
  int nr_lcds;
  struct sensor *s;
  int loop_delay = LOOP_DELAY;
  char owfs_dir[FILENAME_LEN+1];

  last_update.tv_sec = last_update.tv_usec = 0;

  arg = 1;
  while(arg < argc) {
    if(argv[arg][0] == '-') {
      switch(argv[arg][1]) {
      case 'd':
	if(argc <= ++arg) usage(argv);
	loop_delay = atoi(argv[arg]);
	break;
      case 'p':
	print_results = 1;
	break;
      case 'b':
	read_buttons = 1;
	break;
      default:
	usage(argv);
      }
    } else {
      usage(argv);
    }
    arg++;
  }

  owfs_dir[0] = '\000';

#ifndef __UCLIBC__
  /* just for testing locally */
  //strcpy(owfs_dir, "/tmp/var/1wire/uncached");
#endif
  
  if(!(found_fuse = find_fusedev(owfs_dir, device_path, uncached_device_path))) {
    printf("Can't find owfs/fuse directory in /proc/mounts\n");
    exit(0);
  }
  printf("fuse-path=[%s] uncached=[%s]\n", device_path, uncached_device_path);

  gettimeofday(&tvnow, NULL);

  sensors = NULL;
#ifndef IGNORE_RRDFILE
  rrd_files = NULL;
#endif
  
#if 0
  // make sure hardcoded LCD is added
  nr_sensors = 0 ;
  add_sensor(uncached_device_path, ADDR_LCD0, NULL, nr_sensors++, 0, NULL, NULL) ;
  add_sensor(uncached_device_path, ADDR_LCD1, NULL, nr_sensors++, 0, NULL, NULL) ;
  nr_sensors += scan_bus(uncached_device_path);

  nr_sensors = 0 ;
  s = sensors;
  while(s) {
    nr_sensors++;
    s = s->next;
  }
#else
  nr_sensors = scan_bus(uncached_device_path);
  if(nr_sensors == 0) {
    printf("No sensors found on the 1-wire bus\n");
    exit(0);
  }
#endif

  if(print_results) {
    printf("found %d sensors\n", nr_sensors);
    fflush(stdout);
  }

  nr_lcds = 0;
  /* remove all sensors except LCD-displays */
  s = sensors;
  while(s) {
    //printf("sensor [%s] id=%02X\n", s->id, sensor_id(s));
    if(sensor_id(s) != 0xFF) {
      struct sensor *ss = s->next;
      //printf("free_sensor [%s] id=%02X\n", s->id, sensor_id(s));
      if(print_results) {
	printf("sensor [%s]\n", s->id);
      }
#if 0
      if(free_sensor(s)) nr_sensors--;
#endif
      s = ss;
    } else {
      lcd_set_poweron(s->id, 1);
      lcd_set_backlight(s->id, 0);
      add_message("Magnusson 1wire", s->id, 0, 0);
      s = s->next;
      nr_lcds++;
    }
  }

  if(print_results) {
    printf("%d LCD display%s\n", nr_lcds, (nr_lcds>1?"s":""));
    fflush(stdout);
  }

  nr_errors = 0;
  while(1) {
    if(!(found_fuse = find_fusedev(owfs_dir, device_path, uncached_device_path))) {
      printf("Error: Can't find fuse mount anymore\n");
      nr_errors++;
      my_delay(loop_delay*1000);
      continue;
    }
    //if(first_loop) add_message("Magnusson 1wire", ACTIVE_LCD, 0, 0);
  
    if(found_fuse) {
      if(gettimeofday(&tvnow, NULL) < 0) {
	printf("main: gettimeofday %s\n", strerror(errno));
      }
      s = sensors;
      while(s) {
#if 0
	if(first_loop) {
	  if(print_results) {
	    printf("sensor [%s]\n", s->id);
	    fflush(stdout);
	  }
	}
#endif
	if(sensor_id(s) == 0xFF) {
	  if(read_buttons) {
	    if((rc = read_button(s)) < 0) {
	      //printf("Error reading: %s\n", s->id);
	      if(rc == -ENOTCONN) nr_errors = MAX_ERRORS;
	      s = s->next;
	      continue;
	    }
	  }
	  nr_errors = 0;
	  
	  if(lcd_get_backlight(s->id) == 1) {
	    struct timeval *tv = lcd_get_last_button( s->id);
	    
	    if(tv && ((tv->tv_sec+LCD_BACKLIGHT_TIMEOUT) < tvnow.tv_sec)) {
	      // turn off backlight
	      lcd_set_backlight(s->id, 0);
	    }
	  }
	  for(i=0; i<LCD_ROWS; i++) {
	    struct mess *m;
	    m = get_message(s->id, i, TIMED_MESSAGE);
	    if(m) {
	      if(timeval_diff(NULL, &tvnow, &m->end_time) >= 0) {
		delete_message(m);  // delete exipired message
		// new timed message?
		m = get_message(s->id, i, TIMED_MESSAGE);
		if(m) {
		  //printf("main: change to new timed message\n");
		  print_message(m);
		} else {
		  m = get_message(s->id, i, STATIC_MESSAGE);
		  if(m) {
		    // static message
		    //printf("main: change to static message\n");
		    print_message(m);
		  }
		}
	      }
	    }
	  }
//#define TIMEINTERVALL 10
#if 0
	  if((tvnow.tv_sec % 8) < 4) {
	    if(!time_printed && ((tvnow.tv_sec - last_update.tv_sec) > 0)) {
	      print_temp();
	      last_update.tv_sec = tvnow.tv_sec;
	      time_printed = 1;
	    }
	  } else {
	    if((tvnow.tv_sec - last_update.tv_sec) > 0) {
	      print_time();
	      last_update.tv_sec = tvnow.tv_sec;
	      time_printed = 0;
	    }
	  }
#endif
	} else {
	  /* I don't care about anything else but LCD display */
	}
	s = s->next;
      }
    }
#ifdef __UCLIBC__
    if(print_results) {
      printf(".");
      fflush(stdout);
    }
#endif
    if(nr_errors >= MAX_ERRORS) {
      // no restart made here...
      nr_errors = 0;
    }
    if(loop_delay > 0) my_delay(loop_delay);
    first_loop = 0;
  }
}
